/*
 * OS Geiger-counter testing and experimentation - Platform: Arduino Mega
 * License: Creative Commons: https://creativecommons.org/licenses/by/3.0/us/
 * LM: 2021
 */

#include <EEPROM.h>                             // Save/restore calibration data
#include <Adafruit_GFX.h>                       // 1.5 inch color OLED
#include <Adafruit_SSD1351.h>
#include <SPI.h>

// User preference settings
// Counts per minute and dose require a minimum of one minute after startup or restart.
// Setting the following parameter true will suppress CPM and dose display during this time.
const boolean SUPPRESS_TOP_DATA_PARTIAL_CPM = false;

// Test modes                                   // [Several removed after testing]
#define RFC1201                                 // Comment-out for textual data output to serial monitor

// Global settings
#define SERIAL_BAUD 115200                      // Adjust as needed
#define CLICK 18                                // Interrupt pin for Geiger click event

// Radio Shack RS27801 / Parallax 5-position switch
#define RIGHT 22                                // Switch pin 2
#define DOWN 23                                 // Switch pin 3
#define LEFT 24                                 // Switch pin 4
#define CENTER 26                               // Switch pin 6
#define UP 27                                   // Switch pin 7

#define PB0 UP                                  // Aliases for shortcut button-presses
#define PB1 DOWN

// Screen dimensions
#define SCREEN_WIDTH  128
#define SCREEN_HEIGHT 128
#define STATUS_ROW 120
#define LEFT_EDGE 0
#define TOP_EDGE 0

// SPI
#define MOSI_PIN 4                              // AKA DIN
#define SCLK_PIN 5                              // Flip DIN and CLK in order to match device pin order 
#define CS_PIN   6                              // Ditto CS and DC
#define DC_PIN   7
#define RST_PIN  8

// Instantiate
Adafruit_SSD1351 display = Adafruit_SSD1351(SCREEN_WIDTH, SCREEN_HEIGHT, CS_PIN, DC_PIN, MOSI_PIN, SCLK_PIN, RST_PIN);  

// Colors
#define BLACK           0x0000
#define BLUE            0x001F
#define RED             0xF800
#define ORANGE          0xFC42
#define GREEN           0x07E0
#define CYAN            0x07FF
#define MAGENTA         0xF81F
#define YELLOW          0xFFE0  
#define WHITE           0xFFFF

#define TEXT_BG BLACK
#define TEXT_FG BLUE

#define BITMAP_BG BLACK
#define GRAPHIC_BG BLACK

#define TOP_DATA_BOX ORANGE
#define HISTOGRAM_BAR YELLOW
#define STATUS_TEXT RED
#define GRAPH_AXIS RED
#define GRAPH_SCALE_TEXT RED
#define MENU_TEXT_COLOR ORANGE
#define MENU_TEXT_HIGHLIGHT GREEN
#define WAIT_MESSAGE_COLOR YELLOW

// Real-time data display (top rectangle)
#define TOP_DATA_X1 4
#define TOP_DATA_X2 64
#define TOP_DATA_Y1 2
#define TOP_DATA_Y2 22
#define TOP_DATA_Y3 12
#define TOP_DATA_TEXT_SIZE 2
#define TOP_DATA_WIDTH 127
#define TOP_DATA_HEIGHT 42
#define TOP_LEGEND_X 52
#define TOP_LEGEND_Y 2

// Optional wait message
#define WAIT_MESSAGE_X 4
#define WAIT_MESSAGE_Y 16
#define WAIT_MESSAGE_TEXT_SIZE 1

// Radiation bitmap
#include "Radiation_warning.c"                  // May cause OLED burn-in!

int16_t htW = 64;                               // Center of bottom part of screen
int16_t htH = 60;
int16_t htX = (128-htW)/2;
int16_t htY = 112-htH;

// Coin toss bitmaps
#include "Penny_Heads_64.c"
#include "Penny_Tails_64.c"

// Calibration location (below top rectangle)
#define CAL_DATA_X 20
#define CAL_DATA_Y 50
#define CAL_DATA_TEXT_SIZE 2
#define CAL_DATA_CHAR_WIDTH 12

// Calibration constants and variables
// Note: Precision is greatly exaggerated - Calibrate tube by measurement of known source
const float TUBE_FACTOR = .00812037037037;      // J305ß - CPM * TUBE_FACTOR = μSv/hour
                                                // For millirems use: 1 mrem = 10 μSv
// Reciprocal of tube factor:
const float CPM_PER_MICROSIEVERTS_PER_HOUR = 123.147092360319;

// 32-bit random number (EEPROM validation key)
const byte CALIBRATION_KEY[] = {0xdf, 0xa7, 0xd5, 0x40};
float tubeFactor;                               // Default or calibrated value
float cpmph;                                    // Edit copy of CPM per μSv per hour

// Time
const unsigned long ONE_MS = 1;
const unsigned long HALFSEC = 500;
const unsigned long ONESEC = 1000;
const unsigned long TWOSEC = 2000;
const unsigned long ONEMIN = 60000;
const int NUMSECS = 60;                         // Seconds per minute
const int NUMMINS = 60;                         // Minutes per hour
const int NUMHRS  = 24;                         // Hours per day
const int NUMBARS = 60;                         // Number of histogram bars in seconds or minutes graph
unsigned long zeroTime;
boolean firstMinute = true;                     // first minute of data acquisition
const char WAIT_MESSAGE[] = "One minute please...";

// Counting
unsigned int clicks[NUMSECS];
unsigned int fifoClicks[NUMSECS];
unsigned int fifoCPM[NUMSECS];
unsigned long fifoCPH[NUMMINS];
unsigned long fifoCPD[NUMHRS];
int16_t scaledPixel[NUMBARS];
int tNdx = 0;
unsigned int iSum = 0;
float sv = 0.;


// Following definitions include redundancies - Numbers are pixels
#define GRAPH_LEFT 10
#define GRAPH_BAR_WIDTH 2
#define GRAPH_BTM 110
#define GRAPH_VRANGE 60
#define GRAPH_HTICK_WIDTH 3
#define GRAPH_VTICK_HEIGHT 3
#define GRAPH_HSCALE 50
#define GRAPH_TEXT_SIZE 1
#define RND_ARRAY_X_INSET 16

// Options menu
const uint16_t MENU_TEXT_SIZE = 1;
const uint16_t MENU_TEXT_X_INSET = 4;           // From left edge of MENU_BOX (see below)
const uint16_t MENU_BOX_BG = 0xffffe6;
const uint16_t MENU_BOX_X = 4;
const uint16_t MENU_BOX_Y = 50;
const uint16_t MENU_BOX_WIDTH = 120;
const uint16_t MENU_BOX_HEIGHT= 70;

const int NUM_LEVELS = 1;                       // Menu implementation is not recursive
const int NUM_ITEMS  = 5;
const char ITEM_1[] = {"Last Minute"};
const char ITEM_2[] = {"Last Hour  "};
const char ITEM_3[] = {"Random Hex "};
const char ITEM_4[] = {"Calibrate  "};
const char ITEM_5[] = {"Return     "};
const int MENU_ITEM_Y_POS[NUM_ITEMS] = {50, 62, 74, 86, 98};

String csh;                                     // Formatted string calibration value (CPM per μSv per hour)
boolean selected_digit_highlighted = false;     // Kludge one-shot
int selected_item = 0;                          // Or 'highlighted' - Selected on button press
int menu_level = 0;                             // Current depth
int cal_item = 0;                               // Analogous to selected_item in calibration context
int selected_digit;                             // Digit selected for editing in calibration context

// Histograms and Table of Random Hex Values
const int CPM_SCALE = 35;                       // Maximum graphed CPM before applying multiplier
const char HIST_SEC[] = "0      -30s     -60s";
const char HIST_MIN[] = "0      -30m     -60m";
const char OPTION_EXIT_MSG[] = "Press CTR to exit.";
const char RND_HEX_TITLE[] = "Random Hex";
const char LONG_DASH[] = "--";
const char ROW_OF_SPACES[] = "          ";

int cpmFactor = 1;                              // When CPM > CPM_SCALE, apply multiplier (divider) to scaled pixels
unsigned int maxCPM;                            // cpmFactor * CPM_SCALE
int cphFactor = 1;                              // Analogous to above, but for counts per hour
unsigned long maxCPH;
unsigned int clicksPerBar[NUMBARS];             // Most recent moving average values
uint16_t pixelsPerBar[NUMBARS];                 // Scaled moving averages
int hNdx = 0;

// Array of random numbers (hex format)
int bitCount = 0;
const int NUM_RANDOM_BYTES = 26;                // For display: 5 rows of 5 bytes per row (hex format)
byte rndArray[NUM_RANDOM_BYTES];                // Ramdom byte array (menu option)
const int RND_HEX_Y_POS[NUM_ITEMS] = {50, 62, 74, 86, 98};

// Miscellaneous constants and variables
boolean vBit = false;                           // For coin toss
unsigned long lastMicros = 0;
unsigned long lastClick = 0;
unsigned long ms;                               // Scratch for millis()
String sTmp;                                    // Referenced in EEPROM read

// Data Logger
boolean heartbeat_on = false;                   // Enabled by RFC1201 command from computer

void setup() {
  pinMode(CLICK, INPUT);
  pinMode(PB0, INPUT_PULLUP);                   // Normally open pushbutton - Press = Ground
  pinMode(PB1, INPUT_PULLUP);                   // Ditto
  Serial.begin(SERIAL_BAUD);                    // For serial monitor or computer application

  display.begin();
  display.fillScreen(BLACK);
  delay(ONESEC);

  paintSplash();
  delay(TWOSEC);
  display.fillScreen(BLACK);
  delay(HALFSEC);

  attachInterrupt(digitalPinToInterrupt(CLICK), isrClick, RISING);
  zeroTime = millis();
  
#ifndef RFC1201
  Serial.println("Geiger Counter Ready");
#endif

  tubeFactor = getTubeFactor();
  zeroCounters();
  initRndArray();

  drawTopDataBox();
  if (SUPPRESS_TOP_DATA_PARTIAL_CPM)
    oledDisplayWait();
    
  displayRadiationWarning();
}

void loop() {
  ms = millis();
  if ((ms % ONESEC) == 0) {
#ifdef RFC1201
    if (heartbeat_on)
      transmitHeartbeat();
#endif
    incrementSeconds();
    processClicks();
  if (swCenter()) {
    menuLoop();                                 // Sub-loop controls menu navigation
  }
    delay(ONE_MS);                              // Prevent processing more than once in the same second
  }

  if (pb0_pressed()) {                          // 'Last Minute' graph
    updateCpmFactor();
    maxCPM = CPM_SCALE * cpmFactor;
    cpm2scaledPixelArray();
    barGraphScaledPixelArray(HIST_SEC);
  }
  else if (pb1_pressed()) {                     // Coin toss
    displayGraphicHT();
  }

#ifdef RFC1201
  serListen();
#endif
}

void isrClick() {
  if (tNdx < NUMSECS)
    clicks[tNdx]++;
  lastClick = micros();
}

void zeroMinSecCounters() {                     // Reset minute and second counters
  for (int i=0; i<NUMSECS; i++) {
    clicks[i] = 0;
    fifoClicks[i] = 0;
    fifoCPM[i] = 0;
  }  
}

void zeroHrDayCounters() {                      // Reset hour and day counters
  for (int i=0; i<NUMMINS; i++)
    fifoCPH[i] = 0;
  for (int i=0; i<NUMHRS; i++)                  // Not used
    fifoCPD[i] = 0;  
}

void zeroCounters() {                           // Reset counters
  zeroMinSecCounters();
  zeroHrDayCounters();
}

void incrementSeconds() {                       // Increment and zero
  if (++tNdx >= NUMSECS) {
    tNdx = 0;
  }
  clicks[tNdx] = 0;
}

void processClicks() {
  if (ms - zeroTime < ONEMIN)                   // Average is invalid if running less than one minute
    if (SUPPRESS_TOP_DATA_PARTIAL_CPM) {
      return;
    }
  if (firstMinute) {                            // One time only actions
    oledDisplayUnits();
  }
  firstMinute = false;
  updateCPMandDose();
  if (ms % ONEMIN == 0)
    updateCPH();

  oledDisplayCPM(iSum);
  oledDisplaySVH(sv);

  vBit = !vBit;                                 // Random bit harvesting
  boolean tBit = vBit;
  if (lastMicros != lastClick) {
    lastMicros = lastClick;
    if (lastMicros % 8 == 0)
      tBit = !tBit;
#ifndef RFC1201
    Serial.print("  ");  
#endif
    if (tBit) {
      updateRndArray(0);
#ifndef RFC1201
       Serial.print(0);
#endif
    }
    else {
      updateRndArray(1);
#ifndef RFC1201
       Serial.print(1);
#endif
    }
  }

#ifndef RFC1201
  Serial.println();
#endif

}

void updateCPMandDose() {                       // Global variables iSum and sv
  iSum = 0;
  for (int i=0; i<NUMSECS; i++)
    iSum += clicks[i];
#ifndef RFC1201
  Serial.print(iSum);
  Serial.print("  CPM");
#endif
  shiftRightArray(fifoCPM, NUMSECS);
  fifoCPM[0] = iSum;
  sv = svDose(iSum);
#ifndef RFC1201
  Serial.print("  ");
  Serial.print(sv);
  Serial.print(" μSv/hr");  
#endif
}

void updateCPH() {                              // Counts per hour
  shiftRightArray(fifoCPH, NUMMINS);
  fifoCPH[0] = 0;
  for (int i=0; i<NUMSECS; i++)                 // Estimated hourly value based on last full minute
    fifoCPH[0] = fifoCPH[0] + fifoCPM[i];
}

void updateCPD() {                              // Counts per day (Not used)
  shiftRightArray(fifoCPD, NUMHRS);
  fifoCPD[0] = 0;
  for (int i=0; i<NUMHRS; i++)
    fifoCPD[0] = fifoCPD[0] + fifoCPH[i];
}

float svDose(int CPM) {                         // μSv/hour
  if (CPM > 0)
    return tubeFactor * float(CPM);             // Ideally tubeFactor reflects calibration
  else
    return 0.;
}

void drawTopDataBox() {
  display.drawRect (0, 0, TOP_DATA_WIDTH, TOP_DATA_HEIGHT, TOP_DATA_BOX);
}


void oledDisplayUnits() {
  display.setCursor(TOP_DATA_X1,TOP_DATA_Y1);
  display.setTextSize(TOP_DATA_TEXT_SIZE);
  display.setTextColor(TEXT_FG, TEXT_BG);
  display.print("CPM");
  display.setCursor(TOP_LEGEND_X,TOP_LEGEND_Y);
  display.setTextSize(TOP_DATA_TEXT_SIZE);
  display.setTextColor(TEXT_FG, TEXT_BG);
  printLowercaseMu(TEXT_FG, TEXT_BG);           // μ display kludge only works for TOP_DATA_TEXT_SIZE 2
  display.print("Sv/Hr");  
}

void oledDisplayWait() {
  display.setCursor(WAIT_MESSAGE_X,WAIT_MESSAGE_Y);
  display.setTextSize(WAIT_MESSAGE_TEXT_SIZE);
  display.setTextColor(WAIT_MESSAGE_COLOR);
  display.print(WAIT_MESSAGE);
  delay(TWOSEC);
  display.setCursor(WAIT_MESSAGE_X,WAIT_MESSAGE_Y);
  display.setTextColor(BLACK);                  // Erase message
  display.print(WAIT_MESSAGE);  
}

void oledDisplayCPM(int count) {                // CPM part of data row CPM + μSv/hr
  display.setCursor(TOP_DATA_X1,TOP_DATA_Y2);
  display.setTextColor(TEXT_FG, TEXT_BG);
  display.setTextSize(TOP_DATA_TEXT_SIZE);
  if (count < 10)
    display.print(" ");
  if (count < 100)
    display.print(" ");
  display.print(count);
}

void oledDisplaySVH(float sv) {                 // μSv/hr part of data row CPM + μSv/hr
  display.setCursor(TOP_DATA_X2,TOP_DATA_Y2);
  display.setTextColor(TEXT_FG, TEXT_BG);
  display.setTextSize(TOP_DATA_TEXT_SIZE);
  display.print(sv, 2);
}

void oledEraseTopData() {                       // Erase two rows of textual data
  display.setCursor(TOP_DATA_X1,TOP_DATA_Y1);
  display.setTextSize(TOP_DATA_TEXT_SIZE);
  display.setTextColor(TEXT_FG, TEXT_BG);
  display.print(ROW_OF_SPACES);
  display.setCursor(TOP_DATA_X1,TOP_DATA_Y2);
  display.setTextColor(TEXT_FG, TEXT_BG);
  display.print(ROW_OF_SPACES);
}

void oledDisplayStatus(String msg) {
  display.setCursor(LEFT_EDGE,STATUS_ROW);
  display.setTextSize(1);
  display.setTextColor(STATUS_TEXT, TEXT_BG);
  display.print(msg);
}

void oledDisplayStatus(String msg, unsigned long mSec) {
  display.setCursor(LEFT_EDGE,STATUS_ROW);
  display.setTextSize(1);
  display.setTextColor(STATUS_TEXT, TEXT_BG);
  display.print(msg);
  delay(mSec);
  display.setCursor(LEFT_EDGE,STATUS_ROW);
  display.setTextColor(TEXT_BG);
  display.print(msg);
}

void paintSplash() {                            // Confirm OLED orientation (imported function)
  display.setTextSize(2);
  display.setCursor(0,60);
  display.setTextColor(BLUE);
  display.print("ll");
  display.setTextColor(YELLOW);
  display.print("o");
  display.setTextColor(ORANGE);
  display.print("y");
  display.setTextColor(GREEN);
  display.print("d");
  display.setTextColor(RED);
  display.print("m");
  display.setTextColor(WHITE);
  display.print(".");
  display.setTextColor(MAGENTA);
  display.print("n");
  display.setTextColor(ORANGE);
  display.print("e");
  display.setTextColor(CYAN);
  display.print("t");
}

// Shift arrays with explicit dimension passing
void shiftRightArray(int iArray[], int dim) {            // Generic shift - discard rightmost value
  for (int i=dim-1; i>0; i--)
    iArray[i] = iArray[i-1];
}

void shiftRightArray(unsigned long iArray[], int dim) {  // Generic shift - discard rightmost value
  for (int i=dim-1; i>0; i--)
    iArray[i] = iArray[i-1];
}

void displayGraphicHT() {
  boolean tBit = vBit;
  if (lastClick %8 == 0)
    tBit = !tBit;
  eraseRadiationWarning();                      // Main page decoration - See note below
  if (tBit)                                     // Doesn't matter which parity value maps to heads
    display.drawRGBBitmap(htX, htY, Penny_Heads_64, htW, htH);
  else
    display.drawRGBBitmap(htX, htY, Penny_Tails_64, htW, htH);
  delay(ONESEC);
  eraseGraphicHT();
  displayRadiationWarning();                    // Screen space shared by histogram or other data
}

void eraseGraphicHT() {
  display.fillRect (htX, htY, htW, htH, TEXT_BG);
}

int16_t counts2pixels(unsigned int iCount, unsigned int iMax) {
  // Convert one value (not array)
  unsigned long iTmp = (unsigned long) iCount * (unsigned long) GRAPH_VRANGE;
  iTmp /= (unsigned long) iMax;
  return (int16_t) iTmp;
}

void cpm2scaledPixelArray() {
  for (int i=0; i<NUMSECS; i++)
    scaledPixel[i] = counts2pixels(fifoCPM[i], maxCPM);
}

void cph2scaledPixelArray() {
  for (int i=0; i<NUMMINS; i++){
    scaledPixel[i] = counts2pixels(fifoCPH[i], maxCPH);
  }
}


void barGraphScaledPixelArray(String xText) {   // Shortcut from main screen (PB1 pressed)
  eraseRadiationWarning();
  drawHistogram(xText);
  displayVerticalScale(maxCPM);                 // Global maxCPM computed in caller (kludge)
  delay(TWOSEC);
  eraseVerticalScale(maxCPM);
  eraseHistogram(xText);
  displayRadiationWarning();
}

void drawHistogram(String xText) {
  displayHistogramBars();
  displayHistogramYtitle();
  displayHistogramXtitle(xText);  
}

void eraseHistogram(String xText) {
  eraseHistogramYtitle();
  eraseHistogramXtitle(xText);
  eraseHistogramBars();  
}

void displayHistogramBars() {
  for (int i=0; i<NUMBARS; i++) {
    display.fillRect (GRAPH_LEFT+i+i, GRAPH_BTM, GRAPH_BAR_WIDTH, -scaledPixel[i], HISTOGRAM_BAR);
  }  
}

void eraseHistogramBars() {
  for (int i=0; i<NUMBARS; i++) {
    display.fillRect (GRAPH_LEFT+i+i, GRAPH_BTM, GRAPH_BAR_WIDTH, -scaledPixel[i], GRAPHIC_BG);
  }  
}

void displayHistogramXtitle(String xText) {
  display.drawFastHLine (GRAPH_LEFT, GRAPH_BTM+1, 2*NUMSECS, GRAPH_AXIS);
  display.drawFastVLine (GRAPH_LEFT+NUMSECS, GRAPH_BTM, -GRAPH_VTICK_HEIGHT, GRAPH_AXIS);
  display.setCursor(GRAPH_LEFT-2, STATUS_ROW);
  display.setTextSize(GRAPH_TEXT_SIZE);
  display.setTextColor(TEXT_FG, TEXT_BG);
  display.print(xText);
}

void eraseHistogramXtitle(String xText) {
  display.drawFastHLine (GRAPH_LEFT, GRAPH_BTM+1, 2*NUMSECS, GRAPHIC_BG);
  display.drawFastVLine (GRAPH_LEFT+NUMSECS, GRAPH_BTM, -GRAPH_VTICK_HEIGHT, GRAPHIC_BG);
  display.setCursor(GRAPH_LEFT-2, STATUS_ROW);
  display.setTextColor(TEXT_BG);
  display.print(xText);
}

void displayHistogramYtitle() {
  display.drawFastVLine (GRAPH_LEFT-1, GRAPH_BTM, - GRAPH_VRANGE, GRAPH_AXIS);
  display.drawFastHLine (GRAPH_LEFT, GRAPH_BTM-(GRAPH_VRANGE/2 + GRAPH_VRANGE/4), GRAPH_HTICK_WIDTH, GRAPH_AXIS);
  display.drawFastHLine (GRAPH_LEFT, GRAPH_BTM-(GRAPH_VRANGE/2), GRAPH_HTICK_WIDTH, GRAPH_AXIS);
  display.drawFastHLine (GRAPH_LEFT, GRAPH_BTM-(GRAPH_VRANGE/4), GRAPH_HTICK_WIDTH, GRAPH_AXIS);
}

void eraseHistogramYtitle() {
  display.drawFastHLine (GRAPH_LEFT, GRAPH_BTM-(GRAPH_VRANGE/2 + GRAPH_VRANGE/4), GRAPH_HTICK_WIDTH, GRAPHIC_BG);
  display.drawFastHLine (GRAPH_LEFT, GRAPH_BTM-(GRAPH_VRANGE/2), GRAPH_HTICK_WIDTH, GRAPHIC_BG);
  display.drawFastHLine (GRAPH_LEFT, GRAPH_BTM-(GRAPH_VRANGE/4), GRAPH_HTICK_WIDTH, GRAPHIC_BG);
  display.drawFastVLine (GRAPH_LEFT-1, GRAPH_BTM, -GRAPH_VRANGE, GRAPHIC_BG);  
}

void updateCpmFactor() {
  int iMax = 0;
  for (int i=0; i<NUMSECS; i++)
    if (fifoCPM[i] > iMax)
       iMax = fifoCPM[i];
  cpmFactor = iMax / CPM_SCALE + 1;
}

// Untimed displays - Press Center button to exit
void updateCphFactor() {
  int iMax = 0;
  for (int i=0; i<NUMMINS; i++)
    if (fifoCPH[i] > iMax)
       iMax = fifoCPH[i];
  cphFactor = iMax / (CPM_SCALE * NUMMINS) + 1;
}

void lastMinuteGraph() {                        // Untimed display version (from menu)
  updateCpmFactor();
  maxCPM = CPM_SCALE * cpmFactor;
  cpm2scaledPixelArray();
  eraseMenuOptions();
  oledDisplayStatus(OPTION_EXIT_MSG, TWOSEC); 
  drawHistogram(HIST_SEC);
  displayVerticalScale(maxCPM);
  while (!swCenter())
    ;
  eraseVerticalScale(maxCPM);
  eraseHistogram(HIST_SEC);
  displayMenuOptions();
}

void lastHourGraph() {
  updateCphFactor();
  maxCPH = CPM_SCALE * NUMMINS * cphFactor;
  cph2scaledPixelArray();
  eraseMenuOptions();
  oledDisplayStatus(OPTION_EXIT_MSG, TWOSEC); 
  drawHistogram(HIST_MIN);
  displayVerticalScale(maxCPH);
  while (!swCenter())
    ;
  fifoCPH[0] = 0;                               // Mark interruption of unspecified duration
  eraseVerticalScale(maxCPH);
  eraseHistogram(HIST_MIN);
  displayMenuOptions();
}

void displayVerticalScale(int maxData) {
  display.setTextColor(GRAPH_SCALE_TEXT);
  display.setCursor(GRAPH_HSCALE, GRAPH_BTM - GRAPH_VRANGE);
  display.print(LONG_DASH);
  display.print(maxData);
  display.print(LONG_DASH);
  display.setCursor(GRAPH_HSCALE, GRAPH_BTM - (GRAPH_VRANGE/2));
  display.print(LONG_DASH);
  display.print(maxData/2);
  display.print(LONG_DASH);
}

void eraseVerticalScale(int maxData) {
  display.setTextColor(TEXT_BG);
  display.setCursor(GRAPH_HSCALE, GRAPH_BTM - GRAPH_VRANGE);
  display.print(LONG_DASH);
  display.print(maxData);
  display.print(LONG_DASH);
  display.setCursor(GRAPH_HSCALE, GRAPH_BTM - (GRAPH_VRANGE/2));
  display.print(LONG_DASH);
  display.print(maxData/2);
  display.print(LONG_DASH);
}

void displayRadiationWarning() {
  display.drawRGBBitmap(htX, htY, Radiation_warning_64x60, htW, htH);  
}

void eraseRadiationWarning() {
  display.fillRect (htX, htY, htW, htH, GRAPHIC_BG);
}

// For menu navigation, etc.
unsigned int swState() {                        // Not used, but S.T.E.T.
  // Detection priority order C, L, R, U, D
  // Value returned           1, 2, 3, 4, 5
  // No press return          0
  if (swCenter())
    return 1;
  else if (swLeft())
    return 2;
  else if (swRight())
    return 3;
  else if (swUp())
    return 4;
  else if (swDown())
    return 5;
  else return 0;
}

// Button presses
boolean pb0_pressed() {
 if (digitalRead(PB0) == HIGH)
   return false;
 while (digitalRead(PB0) == LOW)
   ;
 return true;  
}

boolean pb1_pressed() {
 if (digitalRead(PB1) == HIGH)
   return false;
 while (digitalRead(PB1) == LOW)
   ;
 return true;  
}

boolean swCenter() {
  if (digitalRead(CENTER) == LOW) {
    while (digitalRead(CENTER) == LOW)
      ;
    return true;
  }
  return false;
}

boolean swLeft() {
  if (digitalRead(LEFT) == LOW) {
    while (digitalRead(LEFT) == LOW)
      ;
    return true;
  }
  return false;
}

boolean swRight() {
  if (digitalRead(RIGHT) == LOW) {
    while (digitalRead(RIGHT) == LOW)
      ;
    return true;
  }
  return false;
}

boolean swUp() {
  if (digitalRead(UP) == LOW) {
    while (digitalRead(UP) == LOW)
      ;
    return true;
  }
  return false;
}

boolean swDown() {
  if (digitalRead(DOWN) == LOW) {
    while (digitalRead(DOWN) == LOW)
      ;
    return true;
  }
  return false;
}

void menuLoop() {                               // Replaces loop() for menu navigation
  oledEraseTopData();                           // Count arrays are not real-time in menu context
  initMenuMode();
  menu_level++;
  if (menu_level == 1) {
    selected_item = 5;                          // Default to 'Return' on entry
    displayMenuOptions();
  }
  while (menu_level > 0) {                      // Back to main loop() on exit from menu sub-system
    if (menu_level == 1) {
      if (swCenter()) {
        if (selected_item == 1)
          lastMinuteGraph();
        else if (selected_item == 2)
          lastHourGraph();
        else if (selected_item == 3) {
          displayRndArray();
          eraseRndArray();
          displayMenuOptions();
        }
        else if (selected_item == 4)            // -> Sub-menu
          processCalibration();
        else if (selected_item == 5)            // Exit menus and return to top level
          menu_level--;
      }
      else if (swUp())
        decrementSelectedItem();
      else if (swDown())
        incrementSelectedItem();
    }
    delay(ONE_MS);
  }
  zeroTime = millis();                          // Reset history (Avoid mixing partial minutes)
  zeroMinSecCounters();                         // 'Last hour' will have missing bar indicating interruption
  exitMenuMode();
  if (SUPPRESS_TOP_DATA_PARTIAL_CPM)
    oledDisplayWait();                          // Same as application startup
  firstMinute = true;                           // Data accumulation has been interrupted - Reset flag
  return;
}

void decrementSelectedItem() {
  int previously_selected_item = selected_item;
  selected_item--;
  if (selected_item == 0)
    selected_item = NUM_ITEMS;
  displayMenuItem(previously_selected_item);
  displayMenuItem(selected_item);
}

void incrementSelectedItem() {
  int previously_selected_item = selected_item;
  selected_item++;
  if (selected_item > NUM_ITEMS)
    selected_item = 1;
  displayMenuItem(previously_selected_item);
  displayMenuItem(selected_item);  
}

void initMenuMode() {
  // Clear background
  eraseRadiationWarning();
}

void exitMenuMode() {
  // Restore background
  eraseMenuOptions();
  displayRadiationWarning();
}

void displayMenuOptions() {
  displayMenuItem(1);
  displayMenuItem(2);
  displayMenuItem(3);
  displayMenuItem(4);
  displayMenuItem(5);
}

void eraseMenuOptions() {
  display.fillRect(MENU_BOX_X, MENU_BOX_Y, MENU_BOX_WIDTH, MENU_BOX_HEIGHT, GRAPHIC_BG);
}

void displayMenuItem(int item) {
  // Reverse fg and bg colors if highlighted =OR= use contrasting fg color
  // ID #s are 1, 2, 3, etc. NOT subscripts
  if (item == selected_item)
    display.setTextColor(MENU_TEXT_HIGHLIGHT, TEXT_BG);
  else
    display.setTextColor(MENU_TEXT_COLOR, TEXT_BG);
  display.setTextSize(MENU_TEXT_SIZE);
  display.setCursor(MENU_BOX_X+MENU_TEXT_X_INSET, MENU_ITEM_Y_POS[item-1]);
  if (item == 1)                                // To do: Convert to array
    display.print(ITEM_1);
  else if (item == 2)
    display.print(ITEM_2);
  else if (item == 3)
    display.print(ITEM_3);
  else if (item == 4)
    display.print(ITEM_4);
  else if (item == 5)
    display.print(ITEM_5);
}

void displayMenuItemTitle(String sTitle) {
  display.setCursor(TOP_DATA_X1, TOP_DATA_Y3);
  display.setTextSize(TOP_DATA_TEXT_SIZE);
  display.setTextColor(TEXT_FG, TEXT_BG);
  display.print(sTitle);
}

void eraseMenuItemTitle(String sTitle) {
  display.setCursor(TOP_DATA_X1, TOP_DATA_Y3);
  display.setTextSize(TOP_DATA_TEXT_SIZE);
  display.setTextColor(TEXT_BG);
  display.print(sTitle);  
}

void displayRndArray() {
  String t = "";
  for (int i=0; i<NUM_RANDOM_BYTES; i++)
    t.concat(byte2hex(rndArray[i]) + ' ');
  eraseMenuOptions();
  displayMenuItemTitle(RND_HEX_TITLE);
  display.setTextColor(TEXT_FG, TEXT_BG);
  display.setTextSize(MENU_TEXT_SIZE);
  display.setCursor(MENU_BOX_X+MENU_TEXT_X_INSET+RND_ARRAY_X_INSET, RND_HEX_Y_POS[0]);
  display.print(t.substring(3,18));             // Display 5 bytes per row
  display.setCursor(MENU_BOX_X+MENU_TEXT_X_INSET+RND_ARRAY_X_INSET, RND_HEX_Y_POS[1]);
  display.print(t.substring(18,33));
  display.setCursor(MENU_BOX_X+MENU_TEXT_X_INSET+RND_ARRAY_X_INSET, RND_HEX_Y_POS[2]);
  display.print(t.substring(33,48));
  display.setCursor(MENU_BOX_X+MENU_TEXT_X_INSET+RND_ARRAY_X_INSET, RND_HEX_Y_POS[3]);
  display.print(t.substring(48,63));
  display.setCursor(MENU_BOX_X+MENU_TEXT_X_INSET+RND_ARRAY_X_INSET, RND_HEX_Y_POS[4]);
  display.print(t.substring(63,78));
  oledDisplayStatus(OPTION_EXIT_MSG, TWOSEC);
  while(!swCenter())                            // Center press to return to menu
    ;
  eraseMenuItemTitle(RND_HEX_TITLE);
}

void eraseRndArray() {
  display.fillRect(0, MENU_BOX_Y, 128, MENU_BOX_HEIGHT, TEXT_BG);
}

// Calibration sub-menu
void displayCalibrationHeader() {
  display.setCursor(TOP_DATA_X1,TOP_DATA_Y3);
  display.setTextSize(TOP_DATA_TEXT_SIZE);
  display.setTextColor(TEXT_FG, TEXT_BG);
  display.print("CPM/");                        // One-off title
  printLowercaseMu(TEXT_FG, TEXT_BG);
  display.print("Sv/Hr");  
}

void eraseCalibrationHeader() {
  display.setCursor(TOP_DATA_X1,TOP_DATA_Y3);
  display.setTextColor(TEXT_FG, TEXT_BG);
  display.setTextSize(TOP_DATA_TEXT_SIZE);
  display.print(ROW_OF_SPACES);  
}

void displayCalibrationData() {
  display.setCursor(CAL_DATA_X,CAL_DATA_Y);
  if (cal_item == 0)
    display.setTextColor(MENU_TEXT_HIGHLIGHT, TEXT_BG);
  else
    display.setTextColor(MENU_TEXT_COLOR, TEXT_BG);
  display.setTextSize(CAL_DATA_TEXT_SIZE);
  display.print(csh);
}

void eraseCalibrationData() {
  display.setCursor(CAL_DATA_X,CAL_DATA_Y);
  display.setTextColor(TEXT_BG);
  display.setTextSize(CAL_DATA_TEXT_SIZE);
  display.print(csh);
}

void highlightSelectedDigit() {                 // Inverse color
  display.setCursor(selected_digit * CAL_DATA_CHAR_WIDTH + CAL_DATA_X, CAL_DATA_Y);  display.setTextColor(TEXT_FG, TEXT_BG);
  display.setTextSize(CAL_DATA_TEXT_SIZE);
  display.print(csh.charAt(selected_digit));
  selected_digit_highlighted = true;
}

void unHighlightSelectedDigit() {               // Normal color
  display.setCursor(selected_digit * CAL_DATA_CHAR_WIDTH + CAL_DATA_X, CAL_DATA_Y);
  display.setTextColor(MENU_TEXT_HIGHLIGHT, TEXT_BG);
  display.setTextSize(CAL_DATA_TEXT_SIZE);
  display.print(csh.charAt(selected_digit));
  selected_digit_highlighted = false;
}

void displayCalibrationMenu() {
  display.setTextColor(MENU_TEXT_COLOR, TEXT_BG);
  display.setTextSize(MENU_TEXT_SIZE);
  display.setCursor(MENU_BOX_X+MENU_TEXT_X_INSET, MENU_ITEM_Y_POS[3]);
  if (cal_item == 1)
    display.setTextColor(MENU_TEXT_HIGHLIGHT, TEXT_BG);
  else
    display.setTextColor(MENU_TEXT_COLOR, TEXT_BG);
  display.print("Save and return");             // Single use literal
  display.setCursor(MENU_BOX_X+MENU_TEXT_X_INSET, MENU_ITEM_Y_POS[4]);
  if (cal_item == 2)
    display.setTextColor(MENU_TEXT_HIGHLIGHT, TEXT_BG);
  else
    display.setTextColor(MENU_TEXT_COLOR, TEXT_BG);
  display.print("Exit w/o saving");             // Ditto
}

void initCalibration() {
  cal_item = 0;
  cpmph = getCalibration();                     // Refresh from EEPROM stored value =OR=
//cpmph = 1. / tubeFactor;                      // from tube factor (Deprecated - Automatic if not stored)
  displayCalibrationHeader();
  csh = spcFloat2String(cpmph);
  displayCalibrationData();
  displayCalibrationMenu();
}

void refreshCalibration() {
  displayCalibrationData();
  displayCalibrationMenu();  
}

void exitCalibrationScreen() {
  csh = spcFloat2String(cpmph);
  eraseCalibrationData();
  eraseCalibrationHeader();
  displayMenuOptions();
}

void processCalibration() {
  eraseMenuOptions();
  initCalibration();
  oledDisplayStatus("Press CTR when done.", TWOSEC);
  selected_digit = 3;                           // charAt position of digit being edited
  highlightSelectedDigit();
  cpmphEditLoop();
  oledDisplayStatus("Exiting calibration", TWOSEC);
  eraseMenuOptions();
  eraseCalibrationHeader();
  displayMenuOptions();
}

void cpmphEditLoop() {                          // Edit calibration constant: counts per minute per μSv per hour
  // cpmph is a global string, formatted NNNN.NN 
  boolean edit_in_progress = true;
  char c;
  while (edit_in_progress) {                    // Edit loop
    if (cal_item == 0) {
      if (!selected_digit_highlighted)
        highlightSelectedDigit();
      if (swCenter()) {
        cal_item = 1;
        refreshCalibration();
        selected_digit_highlighted = false;
      }
      else if (swLeft()) {                      // Left/right and Up/down depend on physical switch orientation
        unHighlightSelectedDigit();             // Usage assumes silkscreen lettering on switch is rightside up
        selected_digit--;
        if (selected_digit < 0)                 // Constants based on number length and decimal location
          selected_digit = 6;
        else if (selected_digit == 4)
          selected_digit = 3;
        highlightSelectedDigit();
      }
      else if (swRight()) {
        unHighlightSelectedDigit();
        selected_digit++;
        if (selected_digit > 6)
          selected_digit = 0;
        else if (selected_digit == 4)
          selected_digit = 5;
        highlightSelectedDigit();        
      }
      else if (swUp()) {
        c = csh.charAt(selected_digit);
        c++;
        if (c > 57)                             // Rollover 9 to 0, etc.
          c = 48;
        csh.setCharAt(selected_digit, c);
        highlightSelectedDigit();
      }
      else if (swDown()) {
        c = csh.charAt(selected_digit);
        c--;
        if (c < 48)
          c = 57;
        csh.setCharAt(selected_digit, c);
        highlightSelectedDigit();        
      }
    }
    else if (cal_item == 1) {
      if (swCenter()) {
        saveEditedCalibration();
        edit_in_progress = false;
      }
      else if (swUp()) {
        cal_item = 0;
        refreshCalibration();
      }
      else if (swDown()) {
        cal_item = 2;
        refreshCalibration();
      }
    }
    else if (cal_item == 2) {
      if (swCenter()) {
        edit_in_progress = false;
      }
      else if (swUp()) {
        cal_item = 1;
        refreshCalibration();
      }
      else if (swDown()) {
        cal_item = 0;
        refreshCalibration();
      }
    }
  }
}

void saveEditedCalibration() {
  cpmph = csh.toFloat();
  if (cpmph != 0.) {                            // Update real-time dose computation
    tubeFactor = 1. / cpmph;
    storeCalibration(cpmph);                    // And save edited value to EEPROM
  }
  oledDisplayStatus("Saving to EEPROM!", TWOSEC); 
}

String spcFloat2String(float x) {               // Custom format for tube factor edit NNNN.NN
  int k = round(x * 100.);
  String s = String(k);
  while (s.length() < 6)
    s = "0" + s;
  return s.substring(0, 4) + "." + s.substring(4, 6);
}

// Random number harvesting
void initRndArray() {
  for (int i=0; i<NUM_RANDOM_BYTES; i++)
    rndArray[i] = 0;
}

void updateRndArray(byte b) {
  // b is a one-bit byte (value 0 or 1)
  if (bitCount > 7) {
    bitCount = 0;
    shiftRightRndArray();
  }
  rndArray[0] = (rndArray[0] | b) << 1;
  bitCount++;
}

void shiftRightRndArray() {
  for (int i=NUM_RANDOM_BYTES-1; i>0; i--)
    rndArray[i] = rndArray[i-1];
  rndArray[0] = 0;
}

String byte2hex(byte b) {                       // No doubt there is a better way!
  int l = b / 16, r = b % 16;
  String t = "";
  if (l < 10)
    t = String(l);
  else if (l == 10)
    t = "A"; 
  else if (l == 11)
    t = "B"; 
  else if (l == 12)
    t = "C"; 
  else if (l == 13)
    t = "D"; 
  else if (l == 14)
    t = "E"; 
  else if (l == 15)
    t = "F";
  if (r < 10)
    t.concat(String(r));
  else if (r == 10)
    t.concat("A");
  else if (r == 11)
    t.concat("B");
  else if (r == 12)
    t.concat("C");
  else if (r == 13)
    t.concat("D");
  else if (r == 14)
    t.concat("E");
  else if (r == 15)
    t.concat("F");
  return t;
}

#ifndef RFC1201
void printRndArray() {
  // Debug - Call periodically, e.g. once per minute
  Serial.println();
  String t = "";
  for (int i=0; i<NUM_RANDOM_BYTES; i++)
    t.concat(byte2hex(rndArray[i]) + ' ');
  // rndArray[0] may be incomplete - Start display at rndArray[1]
  Serial.println(t.substring(3,18));
  Serial.println(t.substring(18,33));
  Serial.println(t.substring(33,48));
  Serial.println(t.substring(48,63));
  Serial.println(t.substring(63,78));
}
#endif

// EEPROM utilities
boolean isCalibrationStored() {                 // Does EEPROM have valid calibration data?
  for (int j=0; j<4; j++)
    if ((byte) EEPROM.read(j) != CALIBRATION_KEY[j])
      return false;
  return true;
}

void storeCalibration(float newCalibration) {
  EEPROM.put(4, newCalibration);                // Store revised calibration constant at 4
  for (int j=0; j<4; j++)
    EEPROM.write(j, CALIBRATION_KEY[j]);        // Store validation key at 0
}

float getCalibration() {                        // Return stored calibration constant if valid =OR= default
  if (isCalibrationStored()) {
    float x;
    EEPROM.get(4, x);
    return x;
  }
  return CPM_PER_MICROSIEVERTS_PER_HOUR;
}

float getTubeFactor() {                         // Return tube factor equivalent of stored calibration or default
  if (isCalibrationStored())                    // Redundant check
    return 1. / getCalibration();
  return TUBE_FACTOR;                           // Same as unconditional return 1. / getCalibration();
}

#ifdef RFC1201
const byte SOC = '<';
const byte EOC = '>';                           // End of command = double '>'
char RxCmd[80];                                 // Hilltopper CAT interface used character array
char TxCmd[80];
String sR = "";                                 // String equivalent of RxCmd[]
String sX = "";

void serListen() {
  byte data_byte;
  while (Serial.available()) {
    data_byte = Serial.read();
    if (data_byte == SOC)
      sR = "";
    else if (data_byte == EOC)
      processRxCmd();
    else
      sR.concat((char) data_byte); 
  }
}

void processRxCmd() {
  if (sR == "GETCPM")
    transmitCPM(fifoCPM[0]);
  else if (sR == "HEARTBEAT1")
    heartbeat_on = true;
  else if (sR == "HEARTBEAT0")
    heartbeat_on = false;
  else if (sR.substring(0,7) == "SETDATE")
    fakeSetDate();
}

void transmitCPM(unsigned int uCPM) {
  byte b = (byte) uCPM;
  Serial.write((byte) 0);
  Serial.write((byte) b);
}

void transmitHeartbeat() {
  Serial.write((byte) 0x80);
  Serial.write((byte) clicks[tNdx]);
}

void fakeSetDate() {                            // This has no effect on Data Logger 'Time Sync Failed'
  // No-op, but acknowledge
  Serial.write((byte) 0xAA);
}

#endif

// Junk
void printLowercaseMu(uint16_t fgColor, uint16_t bgColor) {
  if (true) {                                   // textSize = 2 ONLY!
    uint16_t x = display.getCursorX();          // x = left edge of 'u' glyph
    uint16_t y = display.getCursorY() + 10;     // y = base of 'u'
    
    display.print("u");                         // Mirroring lowercase 'u' approximates μ

    display.drawPixel(x+2, y,   fgColor);       // Draw left short base segment
    display.drawPixel(x+3, y,   fgColor);
    display.drawPixel(x+2, y+1, fgColor);
    display.drawPixel(x+3, y+1, fgColor);

    display.drawPixel(x+6, y,   bgColor);       // Erase right short base segment
    display.drawPixel(x+7, y,   bgColor);
    display.drawPixel(x+6, y+1, bgColor);
    display.drawPixel(x+7, y+1, bgColor);

    display.drawPixel(x+2, y+2, bgColor);       // Erase left long base segment
    display.drawPixel(x+3, y+2, bgColor);
    display.drawPixel(x+2, y+3, bgColor);
    display.drawPixel(x+3, y+3, bgColor);

    display.drawPixel(x+6, y+2, fgColor);       // Draw right long base segment
    display.drawPixel(x+7, y+2, fgColor);
    display.drawPixel(x+6, y+3, fgColor);
    display.drawPixel(x+7, y+3, fgColor);

    display.drawPixel(x,   y+2, fgColor);       // Draw left stem
    display.drawPixel(x+1, y+2, fgColor);
    display.drawPixel(x,   y+3, fgColor);
    display.drawPixel(x+1, y+3, fgColor);

    display.drawPixel(x+8, y+2, bgColor);       // Erase right stem
    display.drawPixel(x+9, y+2, bgColor);
    display.drawPixel(x+8, y+3, bgColor);
    display.drawPixel(x+9, y+3, bgColor);

    display.drawPixel(x-1, y+2, fgColor);       // Add left tick
    display.drawPixel(x-1, y+3, fgColor);
  }
  else
    display.print("u");                         // Other sizes not implemented
}

