/* 
 * My first Arduino program : LM  Rev.1 - 5-letter groups     
 *                                Rev.2 - Speed control       
 *                                Rev.3 - Mother of All Keyers - MOAK
 *                                1.0.3.1   Pseudo-text generator 
 *                                          keyModes (Electronic Keyer)
 *                                                   (Straight key)
 *                                          LCD Display
 *                                          Run-time option selection
 *                                          Add pseudo-call sign mode
 *                                          Sending practice mode
 *                                1.0.3.2   Add speed adjustment mode ('Select' button)            
 *                                          Splash screen gets version ID from VERSION
 *                                Rev.4   - Omitted from this version
 *                                Rev.5   - Hybrid version - October 2019
 *                                1.0.5.1   Internally generated tone, no bug emulation or connected computer features
 *                                1.0.5.2   Compiled for Teensy - Test i2c LCD
 *                                2.x       NetMOAK versions 
 *                                3.0       Touch screen, English language words, radio terms ...
 *                                          
 * Lloyd Milligan (WA4EFS) December 2017 
 *                                         
 *                                4.0       Port to Teensy 4.1 (April 2023) - Requires TFT (Display+touch)
 *                                          Support for character cell LCD is deprecated
 *                                          Support for hardware speed and tone controls is deprecated
 *                                4.0.1     Teensy 4.1 soft reset
 *                                4.0.2     Echo keyed-in characters in 'Electronic Key' option
 *                                          Improve random seed.
 *                                          Improve pseudo-text formatting
 *                                4.0.3     Echo keyed-in characters in 'Straight key' option
 *                                          Support for keyed control sequence in 'Straight key' mode
 *                                4.0.4     Support straight-key input in Sending practice options
 *                                4.1       'Mother Lode' option - Book excerpts for listening practice
 *                                4.1.1     Cosmetic changes to menu options (colors)
 *                                          Extend horizontal touch range of return button
 *                                4.1.2     Screensaver added - Adapted from TFT bubbles demo
 *                                4.1.2a    Added BMP image display control sequence
 *                                4.1.2b    Added Morse time control sequence
 *                                4.1.3     Generalize control sequence to electronic key context
 *                                          Control sequences are the same for both straight key and paddle
 *                                4.1.4     Eliza
 *                                          Make key contacts INPUT_PULLUP in place of external resistors
 *                                4.1.5     Minor bug fix. Manually selecting 'Listen and Send' inherits
 *                                          straight key sending speed when applicable.
 *                                4.1.6     Forward port Morse time-set feature from MOAK-3.2
 *                                          Tweak selectRandomBookFile() function
 *                                4.1.7     Improve control sequence handling and add sequences:
 *                                          /d Morse annunciate date and /dMMDDYYYY set date
 *                                          /s Morse annunciate speed.
 *                                          Add millisecond delay between screensaver iterations.
 *          
 * Lloyd Milligan (WA4EFS) April 2023 - © CC attribution - https://creativecommons.org/licenses/by/3.0/us/
 * 
 *                                4.2       First V4.2 options implemented:
 *                                              Coin toss (control sequence /b)
 *                                              Graphical accuracy scores in sending practice options
 *                                4.2.1     Control sequence '/G' toggles scoring (graphs) on/off. Default is ON
 *                                4.2.2     Support iambic paddle (iambic keying)
 *                                          Change scoring default to OFF (Request of users)
 *                                          Prevent screensaver from starting if keying at menu level 1
 *                                4.2.3     Bug fix - Clock was not sustained on battery when set from Morse
 *                                          Problem did not present when clock was set from computer
 *
 */

#include "SPI.h"
#include "Adafruit_GFX.h"
#include "Adafruit_ILI9341.h"
#include "XPT2046_Touchscreen.h"
#include <EEPROM.h>
#include <SD.h>
#include <TimeLib.h>

const String VERSION = "4.2.3";
const boolean DISPLAY_TOUCH_POINT = false;  // For debugging

#define TFT_CS 10
#define TFT_DC 9
#define TOUCH_CS 8

Adafruit_ILI9341 tft = Adafruit_ILI9341(TFT_CS, TFT_DC);
XPT2046_Touchscreen ts(TOUCH_CS);

// Teensy 4.1 reset (V4)
#define REBOOT_BUTTON 28  // Long press requests software reset

// Standard colors
#define BLACK 0x0000
#define BLUE 0x001F
#define RED 0xF800
#define GREEN 0x07E0
#define CYAN 0x07FF
#define MAGENTA 0xF81F
#define YELLOW 0xFFE0
#define WHITE 0xFFFF

// Color aliases
#define TFT_BG YELLOW
#define TFT_TXT BLUE  // Text color 1
#define TFT_ALT RED   // Text color 2
#define SLIDER_TEXT BLACK

// Additional colors (RGB)
// https://www.w3schools.com/colors/colors_picker.asp
uint8_t LT_GREY[] = { 224, 224, 224 };
uint8_t TAN[] = { 255, 236, 179 };
uint8_t LT_BROWN[] = {255, 224, 179};
uint8_t BROWN[] = {153, 92, 0};

// Application constant arrays
#include "Data/radioterms.c"
#include "Data/weights.c"
#include "Data/words.c"

// Tone  - Microcontroller generated square wave smoothed externally
#define TONE_OUT 7         //  DIO pin 7 - Generated tone is squarewave (was pin 8 in MOAK V3)
const int minFreq = 200;   //  Minimum tone frequency - Set to lowest tone desired
const int maxFreq = 2000;  //  Maximum tone frequency - Set to hightest tone desired
int selFreq = 600;         //  Selected tone frequency

const int CSPERROW = 3;     // Number of callsigns per display row (reduce or prevent callsign wrap)
const int speedRange = 30;  // Upper range not tested
const int minWordSpacing = 0;
const int maxWordSpacing = 100;
const int freqRange = 1023;
const String WORD_SPACING_NAMES[9] = { "", "normal", "double", "", "long", "", "longer", "", "longest" };

// XPT2046.h calibration
#define X_PIXELS 320
#define Y_PIXELS 240
const float SLOPE_X = .0949;
const float INTERCEPT_X = -20.746;
const float SLOPE_Y = .0702;
const float INTERCEPT_Y = -24.431;

// TFT touch buttons
const uint16_t button_width = 200;
const uint16_t button_height = 20;
const uint16_t button_half_height = 10;
const uint16_t button_x0 = 20;
const uint16_t button_y0 = 10;
const uint16_t button_radius = 5;
const uint16_t more_button_width = 100;
const uint16_t return_button_x = 140;
const uint16_t return_button_width = 80;
const uint16_t bottom_buttons_y = 210;
const uint16_t button_vert_separation = 2;
const uint16_t button_hor_separation = 5;
const uint16_t right_buttons_x0 = button_x0 + button_width + button_hor_separation;
const uint16_t right_buttons_width = 80;

// Horizontal bars (menu level 2) - 'sliders' is misnomer
const uint16_t sliders_xPos = 40;
const uint16_t sliders_y0 = 20;
const uint16_t sliders_yDelta = 70;
const uint16_t sliders_width = 240;
const uint16_t sliders_height = 10;
const uint16_t sliders_pointer_width = 10;
const uint16_t sliders_pointer_height = 30;
const uint16_t sliders_text_delta = 10;
const int LED_TEST = 35;

// MOAK-generated text
#define moakTftOutputColor BLUE
#define moakTftInputColor ORANGE
const uint16_t moakTftTextSize = 2;
const uint16_t moakTftTextYdelta = 20;
const uint16_t moakTftTextXinset = 20;
const int moakTftROWS = 8;  // These two constants refer to the display window for generated text
const int moakTftCOLS = 24;

// Keyer
// Pin 13 is SPI clock (shared by display and touch)
const int XMITPIN = 3;  // Output - Key CPO or external device DIO 3
const int DOTPIN = 4;   // Dot side of key paddle d4 (Was pin 6 in MOAK V3)
const int DASHPIN = 5;  // Dash side of key =or= straight key (Was pin 7 in MOAK V3)
const int STRAIGHT_KEY = DASHPIN;

// Additional MOAK constants
const int MAXWORDLENGTH = 15;  // wlen[] dimension
const int GENWORDLENGTH = 9;   // This value must not exceed MAXWORDLENGTH.
const int NUMALPHA = 26;       // Number of alphabetic characters (letters)
const int NUMVOWELS = 5;       // Number of vowels
const int NUMCONSTS = NUMALPHA - NUMVOWELS;
const int SDELAY = 500;  // Delay between sentences in pseudo-text mode

const int CPCT = 80;    // Percent of pseudo-words followed by comma
const int DPCT = 20;    // Percent of pseudo-words followed by dash
const int QPCT = 10;    // Percent of pseudo-questions [sentences ending in '?']
const int NPCT = 10;    // Percent of pseudo-words that are numeric
const int MAXNLEN = 5;  // Maximum length of numeric pseudo-word (minimum is 1)

// Maximum pseudo-sentence length (number of words - minimum sentence length)
const int MAXSLEN = 12;
const int MINSLEN = 5;

// Same for punctuation delimited pseudo-phrases
const int MAXPLEN = 4;
const int MINPLEN = 2;

// Punctuation (convenience aliases)
const char PERIOD = '.';
const char DOT = PERIOD;
const char QUESTIONMARK = '?';
const char COMMA = ',';
const char DASH = '-';
const char SPACE = ' ';
const char STROKE = '/';
const char EOL = (char)13;
// Append non-Morse characters that can be output as the default '-' (dash)
const String MORSE = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789,.?/-:";

const int NUMOPT = 9;  // Number of main-level options
                       // 1000 common words and Ham radio terms are V3 add-ons

const String optList[NUMOPT] = { "Electronic Key",
                                 "Straight Key",
                                 "5 Alpha Groups",
                                 "5 Alpha-num Grps",
                                 "Rand Pseudo-text",
                                 "AlpNumPseudo-txt",
                                 "Pseudo callsigns",
                                 "Sending Practice",
                                 "Listen and send" };


int currentOption = 0;        // Select button steps through options

// Sending practice options
String keyinThis = "";  // For display and comparison

// Time constants
const unsigned long MILLISEC = 1;
const unsigned long HALFSEC  = 500;
const unsigned long ONESEC  = 1000;
const unsigned long TWOSEC  = 2000;
const unsigned long FIVESEC = 5000;
const unsigned long SPLASH_DURATION = FIVESEC;
const unsigned long LONG_PRESS = ONESEC;      // Minimum button press duration for to be considered a long press
const unsigned long REBOOT_OK = LONG_PRESS;   // Minimum button press duration to invoke software reset
const unsigned long TOUCH_WAIT = 10;          // Experimental debounce constant
const unsigned long MENU_WAIT = 500;          // After updating display
const unsigned long SCREEN_TIMEOUT = 30000;   // Start screensaver after this time at menu-level 0 or 1
                                              // 30000 = 30 seconds 300000 = 5 minutes                                             
const unsigned long IMAGE_DURATION = FIVESEC; // BMP file display time (custom escape sequence)

// Miscellaneous
const String ZERO = "0";
int v3_mode = 0;                    // Analogous to ptMode, pcsMode, etc. for options added in version 3
int v4_mode = 0;                    // Ditto for options added in version 4 (Kludge)
int tftRow, tftCol;                 // Prior to MOAK-V4 row and col were locally scoped in tftTickerAddCharacter()
boolean unprocessed_touch = false;  // Trickle a single touch through multiple returns, until processed

// Micro SD card
Sd2Card card;
SdVolume volume;
SdFile root;

const int chipSelect = BUILTIN_SDCARD;
boolean SD_card_enabled = false;

const int MAX_BOOKS  = 20;    // Maximum number of text files supported for Morse practice
const int MAX_IMAGES = 20;    // 4.1.3     
const int MAX_FILENAME_LENGTH = 64;
const char EXT[] = "txt";     // Type extension of Morse practice files
File selectedFile;            // Text file source for Morse listening practice
String booksDirList[MAX_BOOKS];
String bmpsDirList[MAX_IMAGES];
unsigned int bookCount = 0;   // Actual number of file names registered in booksDirList
unsigned int bmpCount  = 0;   // Actual number of file names registered in bmpsDirList
int ndxSelectedFile;          // booksDirList subscript of selected file
unsigned long bookStart = 0;  // Character position at which body of selected book begins (after preamble)
unsigned long bookEnd = 0;    // Character position at which body of selected book ends (before post-matter)

// Application variables
boolean subLevel = false;
int tftMenuLevel = 0;  // Enum analog
int tftMaxLevel = 2;
int tftLastMenuLevel = -1;
int selWordSpacing = 50;  // Test slider - Range 0 - 100 [Was previously volume control]
int numSpaces = 1;        // Number of Morse spaces to insert between words - Adjustable

// TFT Touch
uint16_t lastX = -1;
uint16_t lastY = -1;

// MOAK-generated text
uint16_t moakTftTextColor;
int moakTftCharCount = 0;  // Ticker
char moakTftText[moakTftROWS][moakTftCOLS];

// Keyer
int timeConstant = 1020;   // Empirically determined value [Was 850 for ATmega328PU]
int sndSpeed = 15;         // Will be revalued as read from Arduino
int lastSpeed = sndSpeed;  // For detecting change in speed setting
int minSpeed = 5;          // Must be > 0 (Prevent ÷ 0 error and extend max speed)
int dotTime;               // Valued in tftSetSpeed()
int dashTime;              // dotTime + dotTime + dotTime;
int charTime;              // dashTime;
int wordTime;              // dashTime + dashTime + dotTime;
unsigned long skSpeed;     // Straight key speed (4.0.3)

// Additional MOAK variables
boolean ptMode = true;         // Pseudo-text mode
boolean keyMode = true;        // Sound-out keyed characters, etc.
boolean pcsMode = false;       // Pseudo-callsign mode
boolean practiceMode = false;  // Practice sending by keying-in displayed text
boolean incSound = false;      // Sound practice mode text in Morse

// Accept and process input from key or paddle
boolean straight = false;      // Straight key mode

// Made-up (arbitrary) word length weights
long wLen[MAXWORDLENGTH] = { 5, 15, 35, 55, 75, 90, 105, 115, 125, 130, 135, 139, 142, 144, 145 };

// Based on lexical data - first letter, letter, vowel, and consonant weights
long flWeight[NUMALPHA] = { 60593L, 126028L, 213114L, 265556L, 301422L, 339834L,
                            379183L, 421634L, 451726L, 463127L, 487414L, 525509L, 593554L, 618349L, 640923L,
                            712575L, 717273L, 765491L, 869532L, 920780L, 942190L, 959583L, 988103L, 989754L,
                            994943L, 1000000L };
long lWeight[NUMALPHA] = { 2865L, 3394L, 4586L, 5946L, 10398L, 11254L, 11920L,
                           13721L, 16418L, 16475L, 16668L, 18118L, 19013L, 21591L, 24314L, 25075L, 25118L,
                           27356L, 29677L, 32982L, 33955L, 34330L, 34927L, 35011L, 35604L, 35636L };
long vWeight[NUMVOWELS] = { 2865L, 7317L, 10014L, 12737L, 13710L };
long cWeight[NUMCONSTS] = { 529L, 1721L, 3081L, 3937L, 4603L, 6404L, 6461L,
                            6654L, 8104L, 8999L, 11577L, 12338L, 12381L, 14619L, 16940L, 20245L,
                            20620L, 21217L, 21301L, 21894L, 21926L };

char alpha[NUMALPHA] = { 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J',
                         'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
                         'Y', 'Z' };
char vowel[NUMVOWELS] = { 'A', 'E', 'I', 'O', 'U' };
char consonant[NUMCONSTS] = { 'B', 'C', 'D', 'F', 'G', 'H', 'J', 'K', 'L',
                              'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'V', 'W', 'X', 'Y', 'Z' };

boolean TEST = false;          // Canned sentence (all letters and period)
boolean incNum = false;        // Include numbers in 5-character groups
                               // or in generated pseudo-text                               
boolean skCalibrated = false;  // True if straight key has been calibrated
boolean screenSaver = false;   // True while screensaver is ON (active)
unsigned long skWordTime;      // Straight key calibration constants are global
unsigned long skBoundary;
unsigned long skMinDot;
unsigned long skEOC;
unsigned long lastTouched = millis();

// Test (Development only)
String testString = "The quick brown fox jumped over the lazy dog's back.";

// Eliza - Cannot be forward-referenced
boolean ElizaMode = false;
String zInput = "";
String lastInput = "";              // P$ in BASIC

// Morse date rendering
const String DOW[] = {"sunday", "monday", "tuesday", "wednesday", "thursday", "friday", "saturday"};
const String MON_NAME[] = {"january", "february", "march", "april", "may", "june",
                          "july", "august", "september", "october", "november", "december" };

// V.4.2 Sending practice scoring
int correctlyKeyed = 0;
int totalKeyed = 0;
boolean scoring = false;            // Control sequence '/G' toggles scoring in sending practice options

void setup() {
  Serial.begin(115200);             // Reserved for debugging
  randomSeed(millis());
  pinMode(LED_BUILTIN, OUTPUT);     // Visual indicator
  pinMode(LED_TEST, OUTPUT);
  pinMode(XMITPIN, OUTPUT);
  pinMode(TONE_OUT, OUTPUT);        // Version 1.0.4.1 (and subsequent) - AF square wave out
  pinMode(DOTPIN, INPUT_PULLUP);
  pinMode(DASHPIN, INPUT_PULLUP);
  pinMode(REBOOT_BUTTON, INPUT_PULLUP);
  digitalWrite(XMITPIN, LOW);       // key up
  digitalWrite(LED_BUILTIN, LOW);   // LED off
  digitalWrite(LED_TEST, HIGH);     // External LED off

  tft.begin();  // Init TFT LCD screen
  tft.setRotation(1);
  tft.fillScreen(ILI9341_BLACK);
  ts.begin();  // Init Touch
  ts.setRotation(3);

  if (card.init(SPI_HALF_SPEED, chipSelect)) {
    SD_card_enabled = true;         // Formatted as FAT32
    loadBookFileList();
    loadImageFileList();
  }

  getTimeFromHost();                // RTC support added in 4.1.2b
  initClock();
  
  initSNR();                        // Eliza
  
  initScreenSaver();                // Arduino TFT bubbles example
  flashLED(3);                      // Decoration (debug)
  tftHome();                        // Displays splash screen
  myDelay(SPLASH_DURATION);         // Interruptible by touch
  initParams();                     // V3 +
  tftUpdateScreen();
}

void loop() {
  tftProcessTouch();
  if (keyMode) {
    if (tftMenuLevel == 2 && !straight)
      tftEchoPaddleKeyedChar();
    else
      handleKey();
  }
  // Fall through here if NOT keymode
  else if (ptMode) {  // Pseudo-text modes
    handlePtMode();
  }
  else if (pcsMode) {  // Pseudo callsigns
    handlePcsMode();
  }
  else if (practiceMode) {  // Sending practice modes
    v4PracticeSending();
  }
  else if (v3_mode == 1) {
    outputString(rndews());
  }
  else if (v3_mode == 2) {
    outputString(rndrts());
  }
  else if (v4_mode == 1) {
    v4BookOption();
  }
  else if ((currentOption == 2 || currentOption == 3))
    outputString(fiveCharacterGroup() + SPACE);

  if (requestReboot())
    doReboot();
  
  if (screenSaver) {
    oneScreenSaverIteration();
    delay(MILLISEC);
  }
  else if (screensaverOn())
    screenSaver = true;    
}

// Startup initilization

void initParams() {
  varsReadEEPROM();
  tftMenuLevel = 1;
  ptMode = false;
  pcsMode = false;
  incNum = false;
  practiceMode = false;
  keyMode = true;                   // Default
  unprocessed_touch = false;
}

// Display

void tftSplash() {
  tft.fillScreen(TFT_BG);
  tft.setTextColor(TFT_ALT);
  tft.setTextSize(2);
  tft.setCursor(40, 80);
  tft.println("Mother of all Keyers");
  tft.setCursor(35, 100);
  tft.println("    version " + String(VERSION));
  tft.setTextColor(TFT_TXT);
  tft.setCursor(30, 140);
  tft.println("https://www.lloydm.net");
}

void tftHome() {
  tftSplash();
  tftMenuLevel = 0;
}

void setSpeed() {
  tftSetSpeed();
  return;
}

void tftSetSpeed() {
  dotTime = timeConstant / sndSpeed;
  dashTime = dotTime + dotTime + dotTime;
  charTime = dashTime;
  wordTime = dashTime + dashTime + dotTime;
}

void tftNumSpaces() {
  // 'Spaces between words' was previously a volume control, hence the artificial 0 - 100 range
  // Convert to number of spaces between 'words'
  if (selWordSpacing <= 60)
    numSpaces = 1;
  else if (selWordSpacing <= 70)
    numSpaces = 2;
  else if (selWordSpacing <= 80)
    numSpaces = 4;
  else if (selWordSpacing <= 90)
    numSpaces = 6;
  else if (selWordSpacing <= maxWordSpacing)
    numSpaces = 8;
}

void tftDisplayButton(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg, String lbl, uint16_t sz) {
  tft.fillRoundRect(x, y, w, h, button_radius, bg);
  tft.setTextColor(fg);
  tft.setTextSize(sz);
  tft.setCursor(x + 5, y + 2);
  tft.print(lbl);
  tft.drawRoundRect(x, y, w, h, button_radius, BLACK);
  return;
}

void tftDisplayMoreButton() {
  tftDisplayButton(button_x0, bottom_buttons_y, more_button_width, button_height, TFT_ALT, uColor(LT_BROWN), "More...", 2);
}

void tftDisplayReturnButton() {
  tftDisplayButton(return_button_x, bottom_buttons_y, return_button_width, button_height, TFT_ALT, uColor(LT_BROWN), "Return", 2);
}

void tftDisplayEnglishWordsButton() {
  tft.fillRoundRect(right_buttons_x0, button_y0, right_buttons_width, 3 * button_height + 2 * button_vert_separation, button_radius, GREEN);
  tft.drawRoundRect(right_buttons_x0, button_y0, right_buttons_width, 3 * button_height + 2 * button_vert_separation, button_radius, BLACK);
  tft.setTextColor(TFT_TXT);
  tft.setTextSize(moakTftTextSize);
  tft.setCursor(right_buttons_x0 + 5, button_y0 + 2);
  tft.print(" 1000");
  tft.setCursor(right_buttons_x0 + 5, button_y0 + button_height + button_vert_separation);
  tft.print("common");
  tft.setCursor(right_buttons_x0 + 5, button_y0 + 2 * (button_height + button_vert_separation));
  tft.print("words!");
}

void tftDisplayRadioTermsButton() {
  tft.fillRoundRect(right_buttons_x0, button_y0 + 3 * button_height + 2 * button_vert_separation + button_half_height + 2, right_buttons_width, 3 * button_height + 2 * button_vert_separation, button_radius, GREEN);
  tft.drawRoundRect(right_buttons_x0, button_y0 + 3 * button_height + 2 * button_vert_separation + button_half_height + 2, right_buttons_width, 3 * button_height + 2 * button_vert_separation, button_radius, BLACK);
  tft.setTextColor(TFT_TXT);
  tft.setTextSize(moakTftTextSize);
  tft.setCursor(right_buttons_x0 + 5, button_y0 + 3 * (button_height + button_vert_separation) + button_half_height + 2);
  tft.print(" Ham");
  tft.setCursor(right_buttons_x0 + 5, button_y0 + 4 * (button_height + button_vert_separation) + button_half_height);
  tft.print("radio");
  tft.setCursor(right_buttons_x0 + 5, button_y0 + 5 * (button_height + button_vert_separation) + button_half_height);
  tft.print("terms");
}

void tftDisplayBookButton() {
  tft.fillRoundRect(right_buttons_x0, button_y0 + 7*button_height + 5*button_vert_separation + 2, right_buttons_width, 2*(button_height + button_vert_separation), button_radius, uColor(TAN));
  tft.drawRoundRect(right_buttons_x0, button_y0 + 7*button_height + 5*button_vert_separation + 2, right_buttons_width, 2*(button_height + button_vert_separation), button_radius, BLACK);
  tft.setTextColor(uColor(BROWN));
  tft.setTextSize(moakTftTextSize);
  tft.setCursor(right_buttons_x0+5, button_y0 + 7*button_height + 6*button_vert_separation + 2);
  tft.print("Mother");
  tft.setCursor(right_buttons_x0+5, button_y0 + 8*button_height + 7*button_vert_separation);
  tft.print(" Lode");
}


void tftDisplayMenu() {
  tft.fillScreen(TFT_BG);
  for (int i = 0; i < NUMOPT; i++) 
    tftDisplayButton(button_x0, i * (button_height + button_vert_separation) + button_y0, button_width, button_height, TFT_TXT, WHITE, optList[i], 2);
  tftDisplayMoreButton();
  tftDisplayReturnButton();
  tftDisplayEnglishWordsButton();
  tftDisplayRadioTermsButton();
  if (bookCount > 0)
    tftDisplayBookButton(); 
}

void tftDisplaySlider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg, String lbl, uint16_t sz, int iPos, int xLbl) {
  int x0 = iPos, y0 = y - sliders_pointer_height / 2 + h / 2;
  tft.fillRect(x - sliders_pointer_width, y0, w + sliders_pointer_width + sliders_pointer_width, sliders_pointer_height, TFT_BG);
  tft.fillRoundRect(x, y, w, h, button_radius, bg);
  tft.fillRoundRect(x0, y0, sliders_pointer_width, sliders_pointer_height, 2, fg);
  tft.drawRoundRect(x, y, w, h, button_radius, BLACK);
  tft.drawRoundRect(x0, y0, sliders_pointer_width, sliders_pointer_height, 2, BLACK);
  tft.setTextColor(fg);
  tft.setTextSize(sz);
  tft.setCursor(xLbl, y + sliders_pointer_height - h / 2);  // To do: Centering function
  tft.print(lbl);
  return;
}

void tftDisplaySlider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg, String lbl, uint16_t sz, int iPos) {
  tftDisplaySlider(x, y, w, h, fg, bg, lbl, sz, iPos, x + w / 3);
}

void tftDisplaySlider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg, String lbl, uint16_t sz) {
  tftDisplaySlider(x, y, w, h, fg, bg, lbl, sz, x + w / 2);
}

void tftDisplaySlider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg, String lbl) {
  tftDisplaySlider(x, y, w, h, fg, bg, lbl, 2);
}

void tftDisplaySlider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg) {
  tftDisplaySlider(x, y, w, h, fg, bg, "");
}

void tftDisplayTextSpeed() {
  int xPos = (sndSpeed - minSpeed) * 8 + sliders_xPos + sliders_pointer_width;
  const int text_width = 60;
  if (sndSpeed > (minSpeed + speedRange) / 2)
    xPos = xPos - text_width;
  tft.setCursor(xPos + sliders_text_delta, sliders_y0 - sliders_height);
  tft.setTextColor(SLIDER_TEXT);
  tft.setTextSize(1);
  tft.println(String(sndSpeed) + " wpm");
  return;
}

void tftDisplayTextTone() {
  int xPos = (selFreq - minFreq) * sliders_width / freqRange + sliders_xPos + sliders_pointer_width;
  const int text_width = 65;
  if (selFreq > maxFreq / 2)
    xPos = xPos - text_width;
  tft.setCursor(xPos + sliders_text_delta, sliders_y0 + sliders_yDelta - sliders_height);
  tft.setTextColor(SLIDER_TEXT);
  tft.setTextSize(1);
  tft.println(String(selFreq) + " Hz");
  return;
}

void tftDisplayWordSpacingText() {
  int xPos = (selWordSpacing - minWordSpacing) * sliders_width / (maxWordSpacing - minWordSpacing) + sliders_xPos + sliders_pointer_width;
  const int text_width = 88;
  if (selWordSpacing > maxWordSpacing / 2)
    xPos = xPos - text_width;
  tft.setCursor(xPos + sliders_text_delta, sliders_y0 + sliders_yDelta + sliders_yDelta - sliders_height);
  tft.setTextColor(SLIDER_TEXT);
  tft.setTextSize(1);
  //tft.println(String(selWordSpacing) + "%");
  tft.println("(" + String(numSpaces) + ") " + WORD_SPACING_NAMES[numSpaces]);
  return;
}

void tftDisplaySpeedSlider() {
  int iPos = (sndSpeed - minSpeed) * 8 + sliders_xPos;  // Speed range = 5 - 35, slider range = 40 - 280 -> 240 / 30 = 8
  tftDisplaySlider(sliders_xPos, sliders_y0, sliders_width, sliders_height, TFT_TXT, MAGENTA, String(minSpeed) + "      Speed      " + String(minSpeed + speedRange), 2, iPos, 40);
  tftDisplayTextSpeed();
}

void tftDisplayToneSlider() {
  long iPos = (selFreq - minFreq) * sliders_width / freqRange + sliders_xPos;
  tftDisplaySlider(sliders_xPos, sliders_y0 + sliders_yDelta, sliders_width, sliders_height, TFT_TXT, MAGENTA, "Low    Pitch    High", 2, (int)iPos, 40);
  tftDisplayTextTone();
}

void tftDisplayWordSpacingSlider() {
  long iPos = selWordSpacing * sliders_width / maxWordSpacing + sliders_xPos;
  tftDisplaySlider(sliders_xPos, sliders_y0 + sliders_yDelta + sliders_yDelta, sliders_width, sliders_height, TFT_TXT, MAGENTA, "Spaces between words", 2, iPos, 40);
  tftDisplayWordSpacingText();  // This works, but isn't meaningful.
}

void tftSlidersMenu() {
  tft.fillScreen(TFT_BG);
  tftDisplaySpeedSlider();
  tftDisplayToneSlider();
  tftDisplayWordSpacingSlider();
}


void tftUpdateSlider(uint16_t x, uint16_t y) {
  if (tftTouchedInBox(sliders_xPos, sliders_y0, sliders_width, sliders_pointer_height)) {
    // Speed control
    sndSpeed = (x - sliders_xPos) * speedRange / sliders_width + minSpeed;
    if (sndSpeed < minSpeed)
      sndSpeed = minSpeed;
    else if (sndSpeed > (minSpeed + speedRange))
      sndSpeed = minSpeed + speedRange;
    tftSetSpeed();  // dot and dash timing
    displaySpeedSlider();
    varsWriteEEPROM();
  } else if (tftTouchedInBox(sliders_xPos, sliders_y0 + sliders_yDelta, sliders_width, sliders_pointer_height)) {
    // Tone control
    selFreq = (x - sliders_xPos) * freqRange / sliders_width + minFreq;
    if (selFreq < minFreq)
      selFreq = minFreq;
    if (selFreq > (minFreq + freqRange))
      selFreq = minFreq + freqRange;
    displayToneSlider();
    varsWriteEEPROM();
  } else if (tftTouchedInBox(sliders_xPos, sliders_y0 + sliders_yDelta + sliders_yDelta, sliders_width, sliders_pointer_height)) {
    // V3 word spacing control
    selWordSpacing = (x - sliders_xPos) * maxWordSpacing / sliders_width;
    if (selWordSpacing < minWordSpacing)
      selWordSpacing = minWordSpacing;
    else if (selWordSpacing > maxWordSpacing)
      selWordSpacing = maxWordSpacing;
    tftNumSpaces();
    displayWordSpacingSlider();
    varsWriteEEPROM();
  }
}

// TFT Update

void tftUpdateScreen() {
  if (tftMenuLevel < 0)
    tftMenuLevel = 0;
  else if (tftMenuLevel > tftMaxLevel)
    tftMenuLevel = tftMaxLevel;
  if (tftMenuLevel == tftLastMenuLevel)
    return;
  if (tftMenuLevel == 0)
    tftSplash();
  else if (tftMenuLevel == 1)  // Main menu level
    tftDisplayMenu();
  else if ((tftMenuLevel == 2) && (subLevel)) {
    tftSlidersMenu();
  }
  if (tftMenuLevel != 0)
    tftDisplayReturnButton();
  tftLastMenuLevel = tftMenuLevel;
  delay(MENU_WAIT);
}

void tftSetDefaultMode() {
  keyMode = true;
  ptMode = false;
  pcsMode = false;
  practiceMode = false;
  incNum = false;
}

// TFT print

void tftInitTicker() {
  for (int i = 0; i < moakTftROWS; i++) {
    for (int j = 0; j < moakTftCOLS; j++) {
      moakTftText[i][j] = 0;
    }
  }
  tft.fillRect(0, moakTftTextYdelta, X_PIXELS, moakTftTextYdelta * (moakTftROWS + 1), TFT_BG);
  tft.drawRect(moakTftTextXinset / 2, moakTftTextYdelta / 2, X_PIXELS - moakTftTextXinset, moakTftTextYdelta * (moakTftROWS + 1), TFT_ALT);
  tft.setCursor(moakTftTextXinset, moakTftTextYdelta);
  tft.setTextColor(moakTftOutputColor);
  tft.setTextSize(moakTftTextSize);
}

void tftTickerAddCharacter(char c) {  // Analog of LCD function: tickerAddCharacter(char c)
  int maxCount = moakTftROWS * moakTftCOLS;
  if (tftAbort()) return;
  if (practiceMode) return;  // Do not duplicate display in practice mode
  if (++moakTftCharCount > maxCount) {
    // Clear and restart at top (simpler than vertical scrolling)
    moakTftCharCount = 1;
  }
  if (moakTftCharCount == 1) {
    tftInitTicker();
  }
  tftRow = moakTftCharCount / moakTftCOLS;
  tftCol = moakTftCharCount % moakTftCOLS;
  if (tftRow > 0 && tftCol == 1) {  // Text Box alignment
    tft.setCursor(moakTftTextXinset, moakTftTextYdelta * (tftRow + 1));
  }
  moakTftText[tftRow][tftCol] = c;
  tft.print(c);
}

void tftPrintText(String s) {
  tft.fillScreen(TFT_BG);
  tft.setCursor(0, 5);
  tft.setTextColor(TFT_ALT);
  tft.setTextSize(2);
  tft.println(s);
}

void tftPrintText(String s, int16_t y) {
  tft.fillScreen(TFT_BG);
  tft.setCursor(0, y);
  tft.setTextColor(TFT_ALT);
  tft.setTextSize(2);
  tft.println(s);
}

void tftPrintText_NoClear(String s, int16_t y) {
  tft.setCursor(0, y);
  tft.setTextColor(TFT_ALT);
  tft.setTextSize(2);
  tft.println(s);
}

void tftPrintText(String s, int16_t x, int16_t y) {
  tft.fillScreen(TFT_BG);
  tft.setCursor(x, y);
  tft.setTextColor(TFT_ALT);
  tft.setTextSize(2);
  tft.println(s);
}

void tftPrintText_NoClear(String s, int16_t x, int16_t y) {
  tft.setCursor(x, y);
  tft.setTextColor(TFT_TXT);
  tft.setTextSize(2);
  tft.println(s);
}

void tftPrintText_NoClear(String s, int16_t x, int16_t y, int16_t color) {
  // MOAK-5 (Used in calibration option)
  tft.setCursor(x, y);
  tft.setTextColor(color);
  tft.setTextSize(2);
  tft.println(s);
}

// To do: Deprecate next
void displayTFT(String line1, String line2) {
  // TFT analog of: void displayLCD(String line1, String line2)
  // Variable row is not relevant
  tftInitTicker();
  tft.setCursor(moakTftTextXinset, moakTftTextYdelta);
  tft.print(line1);
  tft.setCursor(moakTftTextXinset, 2 * moakTftTextYdelta);
  tft.print(line2);
}

void tftPrintSelectedOption(int iOpt) {
  String s = "No option";
  if (v3_mode == 1) {
    s = "Common English words";
    tftMenuLevel = 2;
  } else if (v3_mode == 2) {
    s = "Ham radio terms";
    tftMenuLevel = 2;
  } else if (iOpt >= 0) {
    s = optList[iOpt];
    tftMenuLevel = 2;
  }
  tftPrintText(" --> " + s, 100);
  tftDisplayReturnButton();
  if ((iOpt != 1) || (v3_mode > 0)) {
    delay(TWOSEC);
    moakTftCharCount = 0;
    tftInitTicker();
  }
}

// Sliders

void displaySlider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg, String lbl, uint16_t sz, int iPos, int xLbl) {
  int x0 = iPos, y0 = y - sliders_pointer_height / 2 + h / 2;
  tft.fillRect(x - sliders_pointer_width, y0, w + sliders_pointer_width + sliders_pointer_width, sliders_pointer_height, TFT_BG);
  tft.fillRoundRect(x, y, w, h, button_radius, bg);
  tft.fillRoundRect(x0, y0, sliders_pointer_width, sliders_pointer_height, 2, fg);
  tft.drawRoundRect(x, y, w, h, button_radius, BLACK);
  tft.drawRoundRect(x0, y0, sliders_pointer_width, sliders_pointer_height, 2, BLACK);
  tft.setTextColor(fg);
  tft.setTextSize(sz);
  tft.setCursor(xLbl, y + sliders_pointer_height - h / 2);  // To do: Centering function
  tft.print(lbl);
  return;
}

void displaySlider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg, String lbl, uint16_t sz, int iPos) {
  displaySlider(x, y, w, h, fg, bg, lbl, sz, iPos, x + w / 3);
}

void displaySlider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg, String lbl, uint16_t sz) {
  displaySlider(x, y, w, h, fg, bg, lbl, sz, x + w / 2);
}

void displaySlider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg, String lbl) {
  displaySlider(x, y, w, h, fg, bg, lbl, 2);
}

void displaySlider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg) {
  displaySlider(x, y, w, h, fg, bg, "");
}

void displayTextSpeed() {
  int xPos = (sndSpeed - minSpeed) * 8 + sliders_xPos + sliders_pointer_width;
  const int text_width = 60;
  if (sndSpeed > (minSpeed + speedRange) / 2)
    xPos = xPos - text_width;
  tft.setCursor(xPos + sliders_text_delta, sliders_y0 - sliders_height);
  tft.setTextColor(SLIDER_TEXT);
  tft.setTextSize(1);
  tft.println(String(sndSpeed) + " wpm");
  return;
}

void displayTextTone() {
  int xPos = (selFreq - minFreq) * sliders_width / freqRange + sliders_xPos + sliders_pointer_width;
  const int text_width = 65;
  if (selFreq > maxFreq / 2)
    xPos = xPos - text_width;
  tft.setCursor(xPos + sliders_text_delta, sliders_y0 + sliders_yDelta - sliders_height);
  tft.setTextColor(SLIDER_TEXT);
  tft.setTextSize(1);
  tft.println(String(selFreq) + " Hz");
  return;
}

void displayWordSpacingText() {
  int xPos = (selWordSpacing - minWordSpacing) * sliders_width / (maxWordSpacing - minWordSpacing) + sliders_xPos + sliders_pointer_width;
  const int text_width = 88;
  if (selWordSpacing > maxWordSpacing / 2)
    xPos = xPos - text_width;
  tft.setCursor(xPos + sliders_text_delta, sliders_y0 + sliders_yDelta + sliders_yDelta - sliders_height);
  tft.setTextColor(SLIDER_TEXT);
  tft.setTextSize(1);
  //tft.println(String(selWordSpacing) + "%");
  tft.println("(" + String(numSpaces) + ") " + WORD_SPACING_NAMES[numSpaces]);
  return;
}

void displaySpeedSlider() {
  int iPos = (sndSpeed - minSpeed) * 8 + sliders_xPos;  // Speed range = 5 - 35, slider range = 40 - 280 -> 240 / 30 = 8
  displaySlider(sliders_xPos, sliders_y0, sliders_width, sliders_height, TFT_TXT, MAGENTA, String(minSpeed) + "      Speed      " + String(minSpeed + speedRange), 2, iPos, 40);
  displayTextSpeed();
}

void displayToneSlider() {
  long iPos = (selFreq - minFreq) * sliders_width / freqRange + sliders_xPos;
  displaySlider(sliders_xPos, sliders_y0 + sliders_yDelta, sliders_width, sliders_height, TFT_TXT, MAGENTA, "Low    Pitch    High", 2, (int)iPos, 40);
  displayTextTone();
}

void displayWordSpacingSlider() {
  long iPos = selWordSpacing * sliders_width / maxWordSpacing + sliders_xPos;
  displaySlider(sliders_xPos, sliders_y0 + sliders_yDelta + sliders_yDelta, sliders_width, sliders_height, TFT_TXT, MAGENTA, "Spaces between words", 2, iPos, 40);
  displayWordSpacingText();  // This works, but isn't meaningful.
}


// Touch

boolean tftAbort() {
  // Minimize time here
  // return ts.touched();
  if (unprocessed_touch)
    return true;  
  else if (ts.touched()) {
    while (ts.touched())
      delay(TOUCH_WAIT);
    unprocessed_touch = true;
    return true;
  }
  else
    return false;
}

void tftProcessTouch() {
  if (!ts.touched() && !unprocessed_touch)
    return;
  tftMainTouchProcessing();
  tftUpdateScreen();
  return;
}

void tftMainTouchProcessing() {
  // Touching any point on the splash screen enters main menu
  int iOpt;
  lastTouched = millis();  
  unprocessed_touch = false;
  if (screenSaver) {
    screenSaver = false;
    randomSeed(millis());           // The randomer the merrier
    tftLastMenuLevel = -1;          // Dummy value forces refresh
    // In case 'touch' was emulated by a key press
    noTone(TONE_OUT);               // Sidetone off
    digitalWrite(XMITPIN, LOW);     // transmit off
    digitalWrite(LED_TEST, HIGH);   // Debug (Off)
    tftUpdateScreen();
    return;
  }
  int x, y;
  if (tftMenuLevel < 1) {
    tftMenuLevel = 1;
    return;
  }
  TS_Point p = ts.getPoint();
  lastX = x = x2px(p.x);
  lastY = y = y2py(p.y);
  if (DISPLAY_TOUCH_POINT)
    displayTouchCoordinates(x, y);
  if (tftMenuLevel == 0) {
    tftMenuLevel++;
  } else if (tftReturnButtonPressed()) {
    // Return button pressed
    v3_mode = 0;
    v4_mode = 0;
    ElizaMode = false;
    if (bookCount > 0)
      selectedFile.close();
    if (straight)
      currentOption = iOpt = 1;
    else
      currentOption = iOpt = 0;
    tftMenuLevel--;
    subLevel = false;
    if (tftMenuLevel < 0)
      tftMenuLevel = 0;
    keyMode = true;
  }
  // Buttons
  else if (tftMenuLevel == 1) {
    randomSeed(millis());  // [Re]init random number generator based on time of touch
    iOpt = tftOptionSelected();
    if (tftEnglishWordsButtonPressed()) {
      v3_mode = 1;
      tftMenuLevel = 2;
      tftProcessSelectedOption(0);
    } else if (tftRadioTermsButtonPressed()) {
      v3_mode = 2;
      tftMenuLevel = 2;
      tftProcessSelectedOption(0);
    } else if (iOpt >= 0) {
      v3_mode = 0;
      tftMenuLevel = 2;
      tftProcessSelectedOption(iOpt);
    }
    else if (tftBookButtonPressed() && bookCount > 0) {
      v3_mode = 0;
      v4_mode = 1;
      tftMenuLevel = 2;
      subLevel = false;
      v4InitBookOption();
    }
    else if (tftMoreButtonPressed()) {
      tftMenuLevel = 2;
      subLevel = true;
    }
  }
  // Sliders
  else if (tftMenuLevel == 2) {
    if (subLevel)
      tftUpdateSlider(x, y);
  }
  if (tftMenuLevel < 2) {
    tftSetDefaultMode();  // No option selected (default = key mode)
  }
  tftUpdateScreen();  // To do: Remove redundant updates
}

void tftClearV2Options() {
  keyMode = false;
  ptMode = false;
  pcsMode = false;
  practiceMode = false;
}

void tftProcessSelectedOption(int iOpt) {
  if (v3_mode > 0) {
    tftClearV2Options();
    tftPrintSelectedOption(0);
  } else {
    tftSetSelectedOptionParameters(iOpt);
    currentOption = iOpt;
    tftPrintSelectedOption(iOpt);
  }
}

boolean tftTouchedInBox(uint16_t x, uint16_t y, uint16_t w, uint16_t h) {
  if (lastX < x) return false;
  if (lastY < y) return false;
  if (lastX > (x + w)) return false;
  if (lastY > (y + h)) return false;
  return true;
}

int tftOptionSelected() {
  int option = -1;
  for (int i = 0; i < NUMOPT; i++) {
    if (tftTouchedInBox(button_x0, i * (button_height + button_vert_separation) + button_y0, button_width, button_height)) {
      option = i;
      break;
    }
  }
  return option;
}

boolean tftEnglishWordsButtonPressed() {
  return tftTouchedInBox(right_buttons_x0, button_y0, right_buttons_width, 3 * button_height + 2 * button_vert_separation);
}

boolean tftRadioTermsButtonPressed() {
  return tftTouchedInBox(right_buttons_x0, button_y0 + 2 * button_height + 2 * button_vert_separation + button_half_height + 2, right_buttons_width, 3 * button_height + 2 * button_vert_separation);
}

boolean tftReturnButtonPressed() {
//return tftTouchedInBox(return_button_x, bottom_buttons_y, return_button_width, button_height);
  return tftTouchedInBox(return_button_x - 5, bottom_buttons_y, return_button_width + 10, button_height);
}

boolean tftMoreButtonPressed() {
  return tftTouchedInBox(button_x0, bottom_buttons_y, more_button_width, button_height);
}

boolean tftBookButtonPressed() {
  // Development testing
  return tftTouchedInBox(right_buttons_x0, button_y0 + 7*button_height + 5*button_vert_separation + 2, right_buttons_width, 2*button_height + 2*button_vert_separation);
}

// Touch point coordinates conversion

int x2px(int x) {
  // Parameter is point.x value from XPT2046 library
  // Function returns estimated x-pixel position between 0 and 320
  return (int)((float)x * SLOPE_X + INTERCEPT_X);
}

int y2py(int y) {
  // Parameter is point.y value from XPT2046 library
  // Function returns estimated y-pixel position between 0 and 240
  return (int)((float)y * SLOPE_Y + INTERCEPT_Y);
}


// Text and pseudo-text

boolean isVowel(char c) {  // true if and only if c is a vowel
  for (int i = 0; i < NUMVOWELS; i++)
    if (c == vowel[i]) return true;
  return false;
}

int rndwl() {  // Return a weighted random word length
  int len;
  long r = (long)random(wLen[GENWORDLENGTH - 1]);
  for (len = 0; len < GENWORDLENGTH; len++)
    if (r < wLen[len]) break;
  return ++len;
}

char rndfl() {  // Random FIRST letter
  int i;
  long r = (long)random(flWeight[NUMALPHA - 1]);
  for (i = 0; i < NUMALPHA; i++)
    if (r < flWeight[i]) {
      return alpha[i];
    }
  return '*';  // Should be unreachable
}

char rndl() {  // Random letter
  int i;
  long r = (long)random(lWeight[NUMALPHA - 1]);
  for (i = 0; i < NUMALPHA; i++)
    if (r < lWeight[i]) return alpha[i];
  return '!';  // Should be unreachable
}

char rndc() {  // Random consonant
  int i;
  long r = (long)random(cWeight[NUMCONSTS - 1]);
  for (i = 0; i < NUMCONSTS; i++)
    if (r < cWeight[i]) return consonant[i];
  return '@';  // Should be unreachable
}

char rndv() {  // Random vowel
  int i;
  long r = (long)random(vWeight[NUMVOWELS - 1]);
  for (i = 0; i < NUMVOWELS; i++)
    if (r < vWeight[i]) return vowel[i];
  return '#';  // Should be unreachable
}

char rndpd() {  // Random phrase delimiter
  // Revise to incorporate additional delimiters as desired
  int r = random(CPCT + DPCT);
  if (r < CPCT)
    return COMMA;
  return DASH;
}

String rndnum() {  // Random numeric string (pseudo-word) satisfying MAXNLEN
  long r = (long)random(pow(10, random(MAXNLEN) + 1)) + 1;
  return String(r);
}

String rndpw() {  // Generate random pseudo-word
  int len = rndwl();
  if (len == 1) {
    if (random(2) == 0) return "I";
    else return "A";
  }
  char chr = rndfl();
  boolean v = isVowel(chr);
  String word = "";
  word.concat(chr);
  for (int i = 2; i <= len; i++) {
    if (i % 3 == 1) {
      chr = rndl();
      v = isVowel(chr);
      word.concat(chr);
      continue;
    } else {
      if (v) word.concat(rndc());
      else word.concat(rndv());
      v = not(v);
    }
  }
  return word;
}

String englishWord() {
  // Pseudo-random word selected from 1000 most common words in written English
  // Probability = occurrence frequency in written English
  long N = sizeof(WORDWEIGHT) / sizeof(WORDWEIGHT[0]);  // WORDLIST and WORDWEIGHT arrays have the same size
  long nwt = random(WORDWEIGHT[N - 1]);
  int ndx;
  for (ndx = 0; ndx < N; ndx++)
    if (nwt < WORDWEIGHT[ndx])
      break;
  return WORDLIST[ndx];
}

String radioTerm() {
  return RADIOTERM[random(NUMTERMS)];
}

String rndps() {  // Random pseudo-sentence
  char punc;
  char lp = PERIOD;
  int sl = random(MAXSLEN) + MINSLEN;
  int pl = random(MAXPLEN) + MINPLEN;
  String s = "";
  if (random(100) < QPCT) lp = QUESTIONMARK;
  for (int i = 1; i < sl; i++) {
    if (tftAbort())
      return s;
    s.concat(rndpw());
    // Phrase punctuation
    if (i % pl == 0) {
      punc = rndpd();
      if (punc == DASH) s.concat(SPACE);
      s.concat(punc);
      pl = random(MAXPLEN) + MINPLEN;  // Phrase lengths should vary within sentence
    }
    s.concat(SPACE);
  }
  s.concat(rndpw());
  s.concat(lp);
  s.concat(SPACE);
  return tftFormat(s);
}

String rndpsn() {  // Random pseudo-sentence including numbers
  char punc;
  char lp = PERIOD;
  int sl = random(MAXSLEN) + MINSLEN;
  int pl = random(MAXPLEN) + MINPLEN;
  String s = "";
  if (random(100) < QPCT) lp = QUESTIONMARK;
  String rn = rndnum();
  String pw;
  for (int i = 1; i < sl; i++) {
    if (random(100) < NPCT) {
      pw = rn;
      rn = rndnum();
    } else pw = rndpw();
    s.concat(pw);
    // Phrase punctuation
    if (i % pl == 0) {
      punc = rndpd();
      if (punc == DASH) s.concat(SPACE);
      s.concat(punc);
      pl = random(MAXPLEN) + MINPLEN;
    }
    s.concat(SPACE);
  }
  s.concat(rndpw());
  s.concat(lp);
  s.concat(SPACE);
  return tftFormat(s);
}

String rndews() {
  String s = "";
  int sl = random(MAXSLEN) + MINSLEN;
  for (int i = 1; i < sl; i++) {
    s.concat(englishWord() + SPACE);
  }
  s.concat(englishWord() + PERIOD);
  return tftFormat(s);
}

String rndrts() {
  String s = "";
  int sl = random(NUMTERMS) + MINSLEN;
  for (int i = 1; i < sl; i++) {
    s.concat(radioTerm() + SPACE);
  }
  s.concat(radioTerm() + PERIOD);
  return tftFormat(s);
}


// Callsigns

String rndpcs() {  // Random pseudo-call sign
  String cs = "";
  if (random(100) < 5) {  // Proportion with a DX prefix before callsign
    cs.concat(rndDXpx());
    cs.concat(STROKE);
  }
  if (random(10) < 4) {  // Proportion of US callsigns
    cs.concat(rndUSpx());
    if (random(10) < 1)  // Proportion of 1-character suffixes (US calls)
      cs.concat(rndAlphaSuffix(1));
    else if (random(10) < 5)  // Proportion of remainder that are 2-letter calls
      cs.concat(rndAlphaSuffix(2));
    else
      cs.concat(rndAlphaSuffix(3));
  } else {
    cs.concat(rndDXpx());
    if (random(10) < 2)  // Proportion of 1-character suffixes
      cs.concat(rndAlphaSuffix(1));
    else if (random(10) < 8)  // Proportion of remainder that are 2-letter calls
      cs.concat(rndAlphaSuffix(2));
    else
      cs.concat(rndAlphaSuffix(3));
  }
  return cs;
}

String rndUSpx() {  // Random US (and territory) prefix
  const char firstLetter[] = "KWAN";
  String prefix = "";
  if (random(10) < 8) {  // Start with K or W
    prefix.concat(firstLetter[random(2)]);
    if (random(10) < 4)  // Append second letter
      prefix.concat(alpha[random(26)]);
  } else {  // Start with A or N (no second letter)
    prefix.concat(firstLetter[random(2) + 2]);
  }
  prefix.concat(random(10));  // Always append a single digit 0 - 9
  return prefix;
}

String rndDXpx() {  // Random DX prefix
  const char firstLetter[] = "BCDEFGHIJLMOPQRSTUVXYZ";
  String prefix = "";
  if (random(10)) {  // Start with a letter
    prefix.concat(firstLetter[random(22)]);
    prefix.concat(random(10));  // Append a single digit
    if ((random(10) < 2) and (prefix.charAt(1) != '0'))
      prefix.concat(random(10));  // And sometimes another
  } else {                        // Start with a number
    prefix.concat(random(9) + 1);
    prefix.concat(alpha[random(26)]);  // Next must be alpha
    prefix.concat(random(10));         // Another numeric digit
  }
  return prefix;
}


String rndAlphaSuffix(int len) {  // Random alpha suffix of specified length
  String suffix = "";
  for (int i = 0; i < len; i++)
    suffix.concat(alpha[random(26)]);
  return suffix;
}

// Keying practice option (Text to be keyed)

void practicePattern() {  // Text to be keyed-in for sending practice
  // Generates one of the option text types
  int r = random(4000);
  if (r < 2000) keyinThis = rndpw();        // Pseudo-word
  else if (r < 3000) keyinThis = rndpcs();  // Pseudo-callsign
  else if (r < 3500) keyinThis = rndnum();  // Numeric string
  else keyinThis = fiveCharacterGroup();    // Five character group
  return;
}

void v4PracticeSending() {  // Display practice text and compare keyed input ...
  // Similar to 'Trainer' function by Tom Lewis http://www.qsl.net/n4tl/
  // Revised in v4 to encapsulate key sensing from either paddle or straight key
  if (tftAbort()) {
    return;
  }
  char c = (char)0;             // Translated character (English character)
  int thisNum = 0, thisDen = 0;
  practicePattern();
  int patternLength = keyinThis.length();  // Length of text to be keyed-in
  int characterPosition = 0;
  boolean success = false;      // Until correctly keyed
  while (not success) {
    if (tftAbort())
      return;
    keyinThis.toUpperCase();    // Cosmetic change for presentation only
    displayTFT(keyinThis, "");
    if (incSound)
      outputString(keyinThis);  // Sound-out text to be keyed-in
    keyinThis.toLowerCase();
    characterPosition = 0;
    while (characterPosition < patternLength) {
      if (tftAbort())
        return;
      if (straight && skCalibrated)  // Either paddle or straight key
        c = skGetChar();
      else
        c = ekGetChar();
      tft.print(c);
      totalKeyed++;             // Increment denominator for scoring 
      thisDen++;
      if (c == keyinThis.charAt(characterPosition)) {
//      tft.print(c);
        correctlyKeyed++;       // Increment numerator for scoring
        thisNum++;
        if (characterPosition++ == patternLength - 1)
          success = true;
      } else {
//      tft.print(c);
        if (tftAbort())
          return;
        myDelay(ONESEC);
        break;
      }
    }  // while character position < length
  }    // while not success (not complete)
  if (scoring) {
    displayNonCumVbar(thisNum, thisDen);
    displayCumVbar(correctlyKeyed, totalKeyed);
    myDelay(TWOSEC);
  }
  else
    myDelay(ONESEC);
}

char ekGetChar() {   // Get one paddle-keyed character
  String mc = "";    // Morse character (dot=1, dash=2)
  int mcph;          // Integer hash of mc
  char c = (char)0;  // Translated character (English character)
  unsigned long minEOC = dashTime / 2;
  long now = millis();
  while (c == (char)0) {
    if (tftAbort())
      return c;
    if (digitalRead(DOTPIN) == LOW) {
      dot();
      mc.concat('1');
      now = millis();
    } 
    if (digitalRead(DASHPIN) == LOW) {    // 4.2.2 Remove 'else' for iambic keying
      dash();
      mc.concat('2');
      now = millis();
    }
    if (mc == "") {
      now = millis();
      continue;
    } else if (millis() - now > minEOC) {
      mcph = pHash(mc);
      mc = "";
      c = decodeMorseChar(mcph);
    }
  }
  return c;
}

char skGetChar() {  // Get one straight-keyed character
  // Helper function - Accessory to v4PracticeSending()
  char c = (char)0;
  String mc = "";  // Morse character
  int mcph;        // Integer hash of mc
  boolean skCleanPress = false;
  unsigned long now = millis();
  unsigned long skLastKeyUp = now;
  unsigned long skDeltaTime;
  while (true) {
    if (tftAbort())
      return c;
    now = millis();
    if (digitalRead(STRAIGHT_KEY) == LOW) {
      skLastKeyUp = millis();   // Strictly speaking, microseconds ago
      tone(TONE_OUT, selFreq);  // Sidetone
      while (digitalRead(STRAIGHT_KEY) == LOW)
        ;
      now = millis();
      noTone(TONE_OUT);  // Sidetone off
      skDeltaTime = now - skLastKeyUp;
      if (skDeltaTime > skBoundary) {
        mc.concat('2');
        skCleanPress = true;
      } else if (skDeltaTime > skMinDot) {
        mc.concat('1');
        skCleanPress = true;
      }
      if (skCleanPress) {
        skLastKeyUp = now;
        skCleanPress = false;  // Reset
      }
    } else if (now - skLastKeyUp > skEOC) {
      if (mc.length() > 0) {
        mcph = pHash(mc);
        mc = "";
        c = decodeMorseChar(mcph);
        return c;
        skLastKeyUp = now;
      }
    }
  }
}

// Cosmetics

String tftFormat(String s) {
  String sL, sR;
  if (s.length() == 0)
    return s;
  while (s.length() < moakTftCOLS)
    s.concat(SPACE);
  int rightmostSpacePos = -1;
  for (int i = moakTftCOLS - 1; i >= 0; i--)
    if (s.charAt(i) == SPACE) {
      rightmostSpacePos = i;
      break;
    }
  sL = s.substring(0, rightmostSpacePos);
  sR = s.substring(rightmostSpacePos + 1);
  while (sL.length() < moakTftCOLS)
    sL.concat(SPACE);
  return sL + tftFormat(sR);
}


// Morse

void handleKey() {  // Dash side of paddle or straight key (but not bug)

  if (straight && (tftMenuLevel == 2)) { // V4
    tftEchoStraightKeyedChar();
    return;
  }

  if (straight) {                       // Straight key
    if (digitalRead(DASHPIN) == LOW) {  // Key DOWN
      tone(TONE_OUT, selFreq);          // Sidetone
      digitalWrite(XMITPIN, HIGH);      // transmit on (if different d-pin than LED)
      digitalWrite(LED_TEST, LOW);      // Debug (On)
      if (screenSaver)                  // Key press emulates 'touch'
        unprocessed_touch = true;       // in screensaver mode
      lastTouched = millis();              
     } else {                           // Key up
      noTone(TONE_OUT);                 // Sidetone off
      digitalWrite(XMITPIN, LOW);       // transmit off
      digitalWrite(LED_TEST, HIGH);     // Debug (Off)
    }
    return;
  }
  // Following is not reached if in straight key mode
  if (digitalRead(DOTPIN) == LOW) {     // Paddle DOT side
    dot();
    if (screenSaver)
      unprocessed_touch = true;         // Exit screensaver if applicable              
    lastTouched = millis();              
  }
  if (digitalRead(DASHPIN) == LOW) {    // 4.2.2 Remove 'else' for iambic keying
    dash();
    if (screenSaver)                    // Ditto (emulate touch)
      unprocessed_touch = true;
    lastTouched = millis();              
  }
}

void dot() {
  if (tftAbort()) return;
  digitalWrite(LED_BUILTIN, HIGH);  // LED on
  digitalWrite(XMITPIN, HIGH);      // key down (if different d-pin from LED)
  tone(TONE_OUT, selFreq);          // V.1.0.4.1 (backed in from V2)
  myDelay(dotTime);
  digitalWrite(LED_BUILTIN, LOW);   // LED off
  digitalWrite(XMITPIN, LOW);       // key up
  noTone(TONE_OUT);
  myDelay(dotTime);
}

void dash() {
  if (tftAbort()) return;
  digitalWrite(LED_BUILTIN, HIGH);  // LED on
  digitalWrite(XMITPIN, HIGH);      // key down
  tone(TONE_OUT, selFreq);          // V.2
  myDelay(dashTime);
  digitalWrite(LED_BUILTIN, LOW);   // LED off
  digitalWrite(XMITPIN, LOW);       // key up
  noTone(TONE_OUT);
  myDelay(dotTime);
}

void outputString(String str) {
  boolean repeatSpace = false;
  char chr;
  int i;
  str.toLowerCase();
  for (i = 0; i < (int)str.length(); i++) {
    if (tftAbort()) return;
    chr = str.charAt(i);
    if (chr == SPACE) {
      if (i > 0)
        if (str.charAt(i - 1) == SPACE && !pcsMode)
          continue;
      if (!repeatSpace)
        for (int j = 0; j < numSpaces; j++)  // Not wrapping in (tftPresent)
          myDelay(wordTime);
      repeatSpace = true;
      tftTickerAddCharacter(SPACE);
    } else {
      if (tftAbort()) return;
      repeatSpace = false;
      outputMorseChar(chr);
      if (tftAbort()) return;
      tftTickerAddCharacter(chr);
    }
  }
}

void outputMorseChar(char chr) {
  switch (chr) {
    case 'a':
      dot();
      dash();
      break;
    case 'b':
      dash();
      dot();
      dot();
      dot();
      break;
    case 'c':
      dash();
      dot();
      dash();
      dot();
      break;
    case 'd':
      dash();
      dot();
      dot();
      break;
    case 'e': dot(); break;
    case 'f':
      dot();
      dot();
      dash();
      dot();
      break;
    case 'g':
      dash();
      dash();
      dot();
      break;
    case 'h':
      dot();
      dot();
      dot();
      dot();
      break;
    case 'i':
      dot();
      dot();
      break;
    case 'j':
      dot();
      dash();
      dash();
      dash();
      break;
    case 'k':
      dash();
      dot();
      dash();
      break;
    case 'l':
      dot();
      dash();
      dot();
      dot();
      ;
      break;
    case 'm':
      dash();
      dash();
      break;
    case 'n':
      dash();
      dot();
      break;
    case 'o':
      dash();
      dash();
      dash();
      break;
    case 'p':
      dot();
      dash();
      dash();
      dot();
      break;
    case 'q':
      dash();
      dash();
      dot();
      dash();
      break;
    case 'r':
      dot();
      dash();
      dot();
      break;
    case 's':
      dot();
      dot();
      dot();
      break;
    case 't': dash(); break;
    case 'u':
      dot();
      dot();
      dash();
      break;
    case 'v':
      dot();
      dot();
      dot();
      dash();
      break;
    case 'w':
      dot();
      dash();
      dash();
      break;
    case 'x':
      dash();
      dot();
      dot();
      dash();
      break;
    case 'y':
      dash();
      dot();
      dash();
      dash();
      break;
    case 'z':
      dash();
      dash();
      dot();
      dot();
      break;
    case '0':
      dash();
      dash();
      dash();
      dash();
      dash();
      break;
    case '1':
      dot();
      dash();
      dash();
      dash();
      dash();
      break;
    case '2':
      dot();
      dot();
      dash();
      dash();
      dash();
      break;
    case '3':
      dot();
      dot();
      dot();
      dash();
      dash();
      break;
    case '4':
      dot();
      dot();
      dot();
      dot();
      dash();
      break;
    case '5':
      dot();
      dot();
      dot();
      dot();
      dot();
      break;
    case '6':
      dash();
      dot();
      dot();
      dot();
      dot();
      break;
    case '7':
      dash();
      dash();
      dot();
      dot();
      dot();
      break;
    case '8':
      dash();
      dash();
      dash();
      dot();
      dot();
      break;
    case '9':
      dash();
      dash();
      dash();
      dash();
      dot();
      break;
    case ',':
      dash();
      dash();
      dot();
      dot();
      dash();
      dash();
      break;
    case '.':
      dot();
      dash();
      dot();
      dash();
      dot();
      dash();
      break;
    case '?':
      dot();
      dot();
      dash();
      dash();
      dot();
      dot();
      break;
    case '/':
      dash();
      dot();
      dot();
      dash();
      dot();
      break;
    default:
      dash();
      dot();
      dot();
      dot();
      dash();
  }
  if (tftAbort()) return;
  myDelay(charTime);
}

char decodeMorseChar(int hcode) {  // Inverse of outputMorseChar()
  switch (hcode) {
    case 12: return 'a';
    case 2111: return 'b';
    case 2121: return 'c';
    case 211: return 'd';
    case 1: return 'e';
    case 1121: return 'f';
    case 221: return 'g';
    case 1111: return 'h';
    case 11: return 'i';
    case 1222: return 'j';
    case 212: return 'k';
    case 1211: return 'l';
    case 22: return 'm';
    case 21: return 'n';
    case 222: return 'o';
    case 1221: return 'p';
    case 2212: return 'q';
    case 121: return 'r';
    case 111: return 's';
    case 2: return 't';
    case 112: return 'u';
    case 1112: return 'v';
    case 122: return 'w';
    case 2112: return 'x';
    case 2122: return 'y';
    case 2211: return 'z';
    case 22222: return '0';
    case 12222: return '1';
    case 11222: return '2';
    case 11122: return '3';
    case 11112: return '4';
    case 11111: return '5';
    case 21111: return '6';
    case 22111: return '7';
    case 22211: return '8';
    case 22221: return '9';
    case 21112: return '-';
    case 21121: return '/';
    case 41: return ',';
    case 42: return '.';
    case 43: return '?';
    default: return '*';
  }
}

boolean isMorse(char c) {
  return MORSE.indexOf(c) >= 0;
}

int pHash(String mc) {  // Punctuation hash
  // Supplement to decodeMorseChar()
  if (mc == "221122") return 41;
  if (mc == "121212") return 42;
  if (mc == "112211") return 43;
  return mc.toInt();
}

String fiveCharacterGroup() {
  byte b;
  char c;
  String s = "";

  for (int i = 0; i < 5; i++) {
    if (incNum) {
      b = random(36);  // Can be changed to < 36 for partial numbers
      if (b < 26) b = 65 + b;
      else b = 22 + b;
    } else b = 65 + random(26);
    c = (char)b;
    s.concat(c);
  }
  return s;
}

void handlePtMode() {
  if (incNum) {
    outputString(rndpsn());
    if (tftAbort()) return;
  } else {
    outputString(rndps());
    if (tftAbort()) return;
  }
  myDelay(SDELAY);
  tftTickerAddCharacter(SPACE);  // Else no space between sentences
  return;
}

void handlePcsMode() {
  // For TFT format CSPERROW per display row
  String s = "";
  for (int i = 0; i < CSPERROW; i++) {
    s.concat(rndpcs());
    s.concat(SPACE);
    if (tftAbort()) return;
  }
  outputString(tftFormat(s));
  myDelay(SDELAY);
  return;
}

// New in MOAK version 4

void tftEchoPaddleKeyedChar() {
  // Modification of practiceSending() function without practice pattern to match.
  if (tftAbort()) return;
  String mc = "";  // Morse character
  int mcph;        // Integer hash of mc
  // Minimum inter-character time (Experimentally tweaked @ 15 WPM)
  unsigned long minEOC = (float)dashTime / 2.5;
  unsigned long minKeyedSpace = dashTime + dotTime + dotTime;
  long now = (unsigned long)millis();  // For comparing elapsed times
  boolean spaceAllowed = false;        // Limit to one space between words
  char c;
  char ekLastChar = SPACE;             // Control sequence detection
  boolean ekControlSeq = false;
  boolean csContext = false;           // Control sequence started (in progress)
  char csID;                           // Control sequence convenience alias
  String csPar = "";                   // Control sequence parameter
while (tftMenuLevel == 2) {
    if (tftAbort())
      return;
    if (spaceAllowed) {
      if (millis() - now > minKeyedSpace) {
        if (tftCol > 0) {
          tftTickerAddCharacter(SPACE);
          if (ElizaMode) {
            zInput.concat(SPACE);
            if (isEOT(zInput)) {
              outputString(sEliza(zInput.toUpperCase()));
//            outputString(" BK ");            
              zInput = "";              
            }
          }         
          ekLastChar = SPACE;
        }
        spaceAllowed = false;
        now = millis();
        if (csContext) {
          csExecuteCode(csID, csPar);
          csPar = "";
          csContext = false;
          spaceAllowed = true;
        }
      }
    }
    if (digitalRead(DOTPIN) == LOW) {
      dot();
      mc.concat('1');
      now = millis();
    }
    if (digitalRead(DASHPIN) == LOW) {    // 4.2.2 Remove 'else' for iambic keying
      dash();
      mc.concat('2');
      now = millis();
    }
    if (millis() - now > minEOC) {        // See preceding comment
      if (mc.length() > 0) {
        mcph = pHash(mc);
        mc = "";
        c = decodeMorseChar(mcph);
        if (ElizaMode) {
          zInput.concat(c);
          if ((c > 96) && (c <= 126) && !ekControlSeq)
            c = c & ~(0x20);       
        }
        tftTickerAddCharacter(c);
        if (csContext  && (c != SPACE)) {
          csPar.concat(c);
        }
        else if (ekControlSeq && ekLastChar == STROKE) {
          csID = c;
          csContext = true;
          ekControlSeq = false;
        }
        else if (ekLastChar == (char) 0 || ekLastChar == SPACE)
          if (c == STROKE)
            ekControlSeq = true;
        ekLastChar = c;
        now = millis();
        spaceAllowed = true;
      }
    }
  }  // while (tftMenuLevel == 2)
}

void tftEchoStraightKeyedChar() {
  // Measure speed on first 12 key-downs. If v's then echo enabled
  const int CAL_KEY_DOWNS = 12;
  unsigned long calTimes[CAL_KEY_DOWNS];
  unsigned long keyDown = 0, totalTime = 0;
  unsigned long skAbsMinKeyDown = 10;
  unsigned long skDeltaTime;
  // Acquire timing samples
  int i = 0;
  while (i < CAL_KEY_DOWNS) {
    if (tftAbort())
      return;
    if (digitalRead(STRAIGHT_KEY) == LOW) {
      keyDown = millis();
      tone(TONE_OUT, selFreq);
      // xmit pin, LED, etc. here
      while (digitalRead(STRAIGHT_KEY) == LOW)
        ;
      noTone(TONE_OUT);
      skDeltaTime = millis() - keyDown;
      if (skDeltaTime < skAbsMinKeyDown)
        continue;
      calTimes[i] = skDeltaTime;
      totalTime += calTimes[i++];
    }
  }
  // Analyze sampled timing
  // 3 letter 'V's consist of 9 dots and 3 dashes or 18 dot times
  unsigned long skDotTime = totalTime / 18;
  unsigned long skDashTime = totalTime / 6;
  skSpeed = timeConstant / skDotTime;
  skBoundary = (skDotTime + skDashTime) / 2;
  // Validate the first CAL_KEY_DOWNS as 3 letter 'V's
  boolean enableEcho = true;
  if (calTimes[0] > skBoundary)
    enableEcho = false;
  else if (calTimes[1] > skBoundary)
    enableEcho = false;
  else if (calTimes[2] > skBoundary)
    enableEcho = false;
  else if (calTimes[3] < skBoundary)
    enableEcho = false;
  else if (calTimes[4] > skBoundary)
    enableEcho = false;
  else if (calTimes[5] > skBoundary)
    enableEcho = false;
  else if (calTimes[6] > skBoundary)
    enableEcho = false;
  else if (calTimes[7] < skBoundary)
    enableEcho = false;
  else if (calTimes[8] > skBoundary)
    enableEcho = false;
  else if (calTimes[9] > skBoundary)
    enableEcho = false;
  else if (calTimes[10] > skBoundary)
    enableEcho = false;
  else if (calTimes[11] < skBoundary)
    enableEcho = false;
  // Feedback calibration valid or not
  char c = 'V';
  if (enableEcho) {
    skCalibrated = true;
    tftPrintText_NoClear("Est. Speed: " + String(skSpeed) + " wpm", 60, 130);
    myDelay(TWOSEC);
    tftInitTicker();
    moakTftCharCount = 0;
    tftRow = 0;
    tftCol = 0;
    tftTickerAddCharacter(c);
    tftTickerAddCharacter(c);
    tftTickerAddCharacter(c);
    c = SPACE;
    tftTickerAddCharacter(c);
  } else {
    tftPrintText_NoClear("Wait 2 secs and retry!", 40, 130);
    myDelay(TWOSEC);  // Minimize stray key-presses in calibration
    tftInitTicker();
    return;
  }
  // Valid calibration - Continue
  String mc = "";  // Morse character
  int mcph;        // Integer hash of mc
  skMinDot = skDotTime / 3;
  skWordTime = skDashTime + skDotTime;
  skEOC = skDashTime / 2;
  unsigned long now = millis();
  unsigned long skLastKeyUp = now;
  boolean skCleanPress = false;
  boolean skControlSeq = false;
  char skLastChar = SPACE;             // Control sequence detection
  boolean csContext = false;           // Control sequence started
  char csID;                           // Control sequence convenience alias
  String csPar = "";                   // Control sequence parameter
  // Main echo loop
  while (tftMenuLevel == 2) {  // Detect and display straight key entry
    if (tftAbort())
      return;
    now = millis();
    if (digitalRead(STRAIGHT_KEY) == LOW) {
      skLastKeyUp = millis();   // Strictly speaking, microseconds ago
      tone(TONE_OUT, selFreq);  // Sidetone
      while (digitalRead(STRAIGHT_KEY) == LOW)
        ;
      now = millis();
      noTone(TONE_OUT);  // Sidetone off
      skDeltaTime = now - skLastKeyUp;
      if (skDeltaTime > skBoundary) {
        mc.concat('2');
        skCleanPress = true;
      } else if (skDeltaTime > skMinDot) {
        mc.concat('1');
        skCleanPress = true;
      }
      if (skCleanPress) {
        skLastKeyUp = now;
        skCleanPress = false;  // Reset
      }
    } else if (now - skLastKeyUp > skWordTime) {
      if (c != SPACE) {
        c = SPACE;
        tftTickerAddCharacter(c);
        if (ElizaMode) {
          zInput.concat(SPACE);
          if (isEOT(zInput)) {
            outputString(sEliza(zInput.toUpperCase()));
//          outputString(" BK ");            
            zInput = "";              
          }
        }          
      // Word space time elapsed and space allowed          
        if (csContext) {
          csExecuteCode(csID, csPar);
          csPar = "";
          csContext = false;
        }
      }
      skLastKeyUp = now;
    }
    else if (now - skLastKeyUp > skEOC) {
      if (mc.length() > 0) {
        mcph = pHash(mc);
        mc = "";
        skLastChar = c;
        c = decodeMorseChar(mcph);
        if (ElizaMode) {
          zInput.concat(c);
          if ((c > 96) && (c <= 126) && !skControlSeq)
            c = c & ~(0x20);       
        }
        tftTickerAddCharacter(c);
        if (csContext  && (c != SPACE)) {
          csPar.concat(c);          // Build control sequence parameter string
        }
        else if (skControlSeq && skLastChar == STROKE) {
          csID = c;
          csContext = true;
          skControlSeq = false;
        } 
        else if (skLastChar == (char)0 || skLastChar == SPACE)
          if (c == STROKE)
            skControlSeq = true;
        skLastKeyUp = now;
      }
    }
  }  // while (tftMenuLevel == 2)
}

void csExecuteCode(char code) {
  // Interpret '/' + character sequences
  // To (temporarily) disable option execution in development context,
  // substitute '~' for control sequence ID in if (code == ID)
  if (code == 'b') {
    lastSpeed = sndSpeed;
    if (straight) {
      sndSpeed = skSpeed;
      setSpeed();
    }
    tossPenny();
    if (straight) {       
      sndSpeed = lastSpeed;
      setSpeed(); 
    }   
  }
  else if (code == 'c') {           // Clear display (and ticker)
    tftInitTicker();
    moakTftCharCount = 0;
    tftRow = 0;
    tftCol = 0;
  }
  else if (code == 'd') {           // Date
    lastSpeed = sndSpeed;
    if (straight) {
      sndSpeed = skSpeed;
      setSpeed();
    }
    outputString(sMorseDate());
    if (straight) {       
      sndSpeed = lastSpeed;
      setSpeed(); 
    }   
  }
  else if (code == 'g') {           // Toggle graphs in practice sending options on/off
    lastSpeed = sndSpeed;
    if (straight) {
      sndSpeed = skSpeed;
      setSpeed();
    }
    toggleScoring();
    if (straight) {       
      sndSpeed = lastSpeed;
      setSpeed(); 
    }   
  } 
  else if (code == 'n') {           // new line
    tftTickerAddCharacter(SPACE);
    while (tftCol > 0)
      tftTickerAddCharacter(SPACE);    
  }
  else if (code == 's') {           // Morse speed
    lastSpeed = sndSpeed;
    if (straight) {
      sndSpeed = skSpeed;
      setSpeed();
    }
    outputString(sMorseSpeed());
    if (straight) {       
      sndSpeed = lastSpeed;
      setSpeed(); 
    }   
  }  
  else if (code == 't') {           // Time
    lastSpeed = sndSpeed;
    if (straight) {
      sndSpeed = skSpeed;
      setSpeed();
    }
    outputString(sMorseTime());
    if (straight) {       
      sndSpeed = lastSpeed;
      setSpeed(); 
    }   
  }  
  else if (code == 'x') {           // Exit from 'echo keying' to menu level 1
    screenSaver = true;             // Simulate context - Will auto-abort
    tftMenuLevel = 1;               // Reset to menu level 1
    ElizaMode = false;
    unprocessed_touch = true;       // Simulate a touch
  }
  else if (code == 'z') {           // Toggle Eliza
    if (ElizaMode) {
      ElizaMode = false;
      zInput = "";
      if (straight) {       
        sndSpeed = lastSpeed;
        setSpeed(); 
      }   
      tftTickerAddCharacter(SPACE);
    }
    else {
      ElizaMode = true;
      lastSpeed = sndSpeed;
      if (straight) {
        sndSpeed = skSpeed;
        setSpeed();
      }      
      outputString(sLine160());
      lastInput = "";            
      zInput = "";
    }
  }
  else if (code == '~') {           // Control sequence '0' - Recalibrate touch 
    calibrateTFT();                 // '~' disables option (under development)
    // On return clear screen -
    tft.fillRect(0, 0, X_PIXELS, Y_PIXELS, TFT_BG);
    tftInitTicker();
    tftDisplayReturnButton();
    moakTftCharCount = 0;
    tftRow = 0;
    tftCol = 0;
  }
  else if (code == '8') {           // Sending Practice option
    int iOpt = 7;                   // Option 8 = index 7
    v3_mode = 0;
    tftMenuLevel = 2;
    tftProcessSelectedOption(iOpt);
    tftUpdateScreen();    
    unprocessed_touch = true;
  }
  else if (code == '9') {           // Listen and send option
    int iOpt = 8;                   // Option 9 = index 8
    v3_mode = 0;
    tftMenuLevel = 2;
      lastSpeed = sndSpeed;
      if (straight) {
        sndSpeed = skSpeed;
        setSpeed();
      }      
    tftProcessSelectedOption(iOpt);
    tftUpdateScreen();    
    unprocessed_touch = true;
  }
  else if (code == '/') {           // Display random BMP file
    char fnamCharArray[MAX_FILENAME_LENGTH];
    String fileName = bmpsDirList[random(bmpCount)];
    tft.fillScreen(BLACK);
    for (int i=0; i<MAX_FILENAME_LENGTH; i++)
      fnamCharArray[i] = (char) 0;
    fileName.toCharArray(fnamCharArray, fileName.length() + 1);
    bmpDraw(fnamCharArray, 0, 0);
	  myPureDelay(IMAGE_DURATION);
    tft.fillScreen(YELLOW);
    tftInitTicker();
    tftDisplayReturnButton();    
    moakTftCharCount = 0;
    tftRow = 0;
    tftCol = 0;
  }
  // Additional sequences here
  else
    tftTickerAddCharacter(SPACE);
  return;
}

void csExecuteCode(char code, String csPar) {
  // Interpret '/' + character sequences
  // that include a parameter (e.g. 'set' sequences)
  if (code == 'd') {
    if (csPar.length() == 0)
      csExecuteCode(code);
    else if (csPar.length() == 8) {
      if (isDigits(csPar)) {
      mySetDate(csPar);
      }
    }
    tftTickerAddCharacter(SPACE);
  }
  else if (code == 't') {
    if (csPar.length() == 0)
      csExecuteCode(code);
    else if (csPar.length() == 4) {
      if (isDigits(csPar)) {
      mySetTime(csPar);
      }
    }
    tftTickerAddCharacter(SPACE);
  }
  else {
    csExecuteCode(code);
  }
  return;
}

// General

void tftSetSelectedOptionParameters(int iOpt) {
  // Cloned from pushbutton function processSelectButton()
  // Process each option type
  // V4 Allow redundant set if no change
  currentOption = iOpt;
  if (iOpt == 0) {  // Electronic keyer
    keyMode = true;
    if (straight) {
      sndSpeed = lastSpeed;
      tftSetSpeed();      
    }
    straight = false;
    ptMode = false;
    pcsMode = false;
    incNum = false;
    practiceMode = false;
    moakTftCharCount = 0;
  } else if (iOpt == 1) {  // Straight key
    keyMode = true;
    straight = true;
    ptMode = false;
    pcsMode = false;
    incNum = false;
    practiceMode = false;
  } else if (iOpt == 2) {  // 5-letter groups with display
    keyMode = false;
    ptMode = false;
    pcsMode = false;
    incNum = false;
    practiceMode = false;
  } else if (iOpt == 3) {  // 5-character alphanumeric groups, with display
    keyMode = false;
    ptMode = false;
    pcsMode = false;
    practiceMode = false;
    incNum = true;
  } else if (iOpt == 4) {  // Pseudo-text, with display
    keyMode = false;
    ptMode = true;
    pcsMode = false;
    practiceMode = false;
    incNum = false;
  } else if (iOpt == 5) {  // Pseudo-text + numbers, with display
    keyMode = false;
    ptMode = true;
    pcsMode = false;
    practiceMode = false;
    incNum = true;
  } else if (iOpt == 6) {  // Pseudo call signs with display
    keyMode = false;
    ptMode = false;
    practiceMode = false;
    pcsMode = true;
  } else if (iOpt == 7) {  // Practice sending by keying-in displayed text
    keyMode = false;
    ptMode = false;
    pcsMode = false;
    practiceMode = true;
    incSound = false;      // Relevant only in sending practice modes
    totalKeyed = 0;        // V.4.2 scoring
    correctlyKeyed = 0;
  } else if (iOpt == 8) {  // Practice sending by keying-in sounded and displayed text
    keyMode = false;
    if (straight && skCalibrated) {
      sndSpeed = skSpeed;
      tftSetSpeed();
    }
    ptMode = false;
    pcsMode = false;
    practiceMode = true;
    incSound = true;  // Sound pattern to copy
    totalKeyed = 0;        // V.4.2 scoring
    correctlyKeyed = 0;
  }
  // Additional options go here
}

// EEPROM

void writeEEPROM(String s) {
  int sLen = s.length();
  if (sLen > EEPROM.length())
    return;
  for (int i = 0; i < sLen; i++)
    EEPROM.write(i, s.charAt(i));
}

String readEEPROM(int sLen) {
  if (sLen > EEPROM.length()) sLen = EEPROM.length();
  String s = "";
  for (int i = 0; i < sLen; i++)
    s.concat((char)EEPROM.read(i));
  return s;
}

void eraseEEPROM(int len) {
  if (len > EEPROM.length()) len = EEPROM.length();
  for (int i = 0; i < len; i++)
    EEPROM.write(i, 0);
}

void varsWriteEEPROM() {
  // Global variables SPEED, TONE FREQUENCY, WORD SPACING
  String s = String(sndSpeed);
  String t = String(selFreq);
  String v = String(selWordSpacing);
  while (s.length() < 2)
    s = ZERO + s;
  while (t.length() < 4)
    t = ZERO + t;
  while (v.length() < 2)
    v = ZERO + v;
  writeEEPROM("@@" + s + t + v);
}

void varsReadEEPROM() {
  String tmp = readEEPROM(10);
  if (tmp.substring(0, 2) == "@@") {
    sndSpeed = tmp.substring(2, 4).toInt();
    lastSpeed = sndSpeed;
    tftSetSpeed();  // dot and dash timing
    selFreq = tmp.substring(4, 8).toInt();
    selWordSpacing = tmp.substring(8, 10).toInt();
    tftNumSpaces();
  }
  else
    setSpeed();                     // New MOAK, nothing in EEPROM - Set to default
  return;
}

// Micro SD card (v4 Book options)

void loadBookFileList() {
  // SD root directory (Files only)
  if (SD_card_enabled) {
    File root = SD.open("/");
    File entry;
    String fileName;
    for (bookCount = 0; bookCount < MAX_BOOKS; bookCount++) {
      entry = root.openNextFile();
      if (!entry)
        break;
      // Skip directories
      if (entry.isDirectory()) {
        bookCount--;
      } else {
        // Validate as .txt
        fileName = entry.name();
        if (fileName.substring(fileName.length() - 4) == ".txt") {
          // Explicit copy to string list
          booksDirList[bookCount] = "";
          for (unsigned int i = 0; i < fileName.length(); i++) {
            booksDirList[bookCount].concat(fileName.charAt(i));
          }
        }
        else
          bookCount--;
      }
      entry.close();
    }
    root.close();    
  }
}

void loadImageFileList() {
  // SD root directory (Files only)
  if (SD_card_enabled) {
    File root = SD.open("/");
    File entry;
    String fileName;
    for (bmpCount = 0; bmpCount < MAX_IMAGES; bmpCount++) {
      entry = root.openNextFile();
      if (!entry)
        break;
      // Skip directories
      if (entry.isDirectory()) {
        bmpCount--;
      } else {
        // Validate as .txt
        fileName = entry.name();
        if (fileName.substring(fileName.length() - 4) == ".bmp") {
          // Explicit copy to string list
          bmpsDirList[bmpCount] = "";
          for (unsigned int i = 0; i < fileName.length(); i++) {
            bmpsDirList[bmpCount].concat(fileName.charAt(i));
          }
        }
          else
            bmpCount--;
      }
      entry.close();
    }
    root.close();    
  }
}

void selectRandomBookFile() {
  for (int i=0; i<10; i++)          // Kludge - First value after refreshing seed is non-random
    (void) random(bookCount);
  ndxSelectedFile = random(bookCount);
}

void initBookParameters() {
  String fileName = booksDirList[ndxSelectedFile];
  char fnamCharArray[MAX_FILENAME_LENGTH];
  char c;
  int paramsLength;
  String sBeginPos = "";
  String sEndPos = "";
  for (int i=0; i<MAX_FILENAME_LENGTH; i++)
    fnamCharArray[i] = (char) 0;
  fileName.toCharArray(fnamCharArray, fileName.length() + 1);

  selectedFile = SD.open(fnamCharArray);

  if (selectedFile == 0)
    return;
  SD.open(selectedFile.name(), FILE_READ);
  selectedFile.seek(0);
  for (;;) {
    c = selectedFile.read();
    if ('0' <= c && c <= '9')
      sBeginPos.concat(c);
    else {
      break;
    }
  }
  for (;;) {
    c = selectedFile.read();
    if ('0' <= c && c <= '9')
      sEndPos.concat(c);
    else {
      break;
    }
  }
  paramsLength = sBeginPos.length() + sEndPos.length() + 2;
  bookStart = sBeginPos.toInt() + paramsLength;
  bookEnd = sEndPos.toInt() + paramsLength;
  // Caller will close file
}

unsigned long randomStartingPosition() {
  return random(bookEnd - bookStart) + bookStart;
}

void v4InitBookOption() {
  char c;
  tftClearV2Options();
  selectRandomBookFile();
  String selectedFilename = booksDirList[ndxSelectedFile];
  initBookParameters();
  String displayBookTitle;
 // Acquire title from file name (convention)
  String displayTitle;
  int fnLen = selectedFilename.length() - 4;
  displayTitle = selectedFilename.substring(0, fnLen);
  const int WRAP = 18;
  const String sSpace = " ";
  int j = fnLen + 1; 
  // Format long book title for display       
  if (fnLen > WRAP) {
    while (j > 0) {
      j = displayTitle.lastIndexOf(sSpace, j-1);
      if (j <= WRAP)
        break;
    }
    if (j > 0) {
      tftPrintText(" --> " + displayTitle.substring(0, j), 100);
      tftPrintText_NoClear(" " + displayTitle.substring(j), 120); 
    }
    else
      tftPrintText(" --> " + displayTitle, 100);
  }
  else
    tftPrintText(" --> " + displayTitle, 100);
  //
  if (selectedFile == 0) {
    tftPrintText(" Book open failed!");
    delay(TWOSEC);  
    v4_mode = 0;
    return;
  }
  else if (bookEnd < 4) {
    tftPrintText(" Parameter error!");
    delay(TWOSEC);
    v4_mode = 0;
    return;
  }  
  unsigned long rndStrtPos = randomStartingPosition();
  selectedFile.seek(rndStrtPos);
  c = SPACE;
  unprocessed_touch = false;        // Kludge - To do: Fix this!
  int MAX_SEARCH_LENGTH = 5000;
  int count = 0;
  while (c != PERIOD) {
    if  (++count > MAX_SEARCH_LENGTH)
      break;
    c = selectedFile.read();
    if (selectedFile.position() >= bookEnd)
      selectedFile.seek(bookStart);
  }
  delay(TWOSEC);
  tftInitTicker();
  moakTftCharCount = 0;
  tftRow = 0;
  tftCol = 0;
}

void v4BookOption() {
  // AKA 'Mother lode'
  char c;
  c = selectedFile.read();
  if (isMorse(c)) {
    outputMorseChar(toLowerCase(c));
    tftTickerAddCharacter(c);
  }
  if (c == SPACE || (byte)c == EOL) {
    for (int j=0; j<numSpaces; j++)
      myDelay(wordTime);
    tftTickerAddCharacter(SPACE);
  }
  if (selectedFile.position() >= bookEnd)
    selectedFile.seek(bookStart);
}


// ------------------------------- Utilities ---------------------------------

boolean isDigits(String s) {        // True iff s consists soley of digits -- nothing else
  for (unsigned int i=0; i<s.length(); i++)
    if (!isDigit(s.charAt(i)))
      return false;
  return true;
}

void flashLED() {
  digitalWrite(LED_TEST, LOW);
  delay(ONESEC / 8);
  digitalWrite(LED_TEST, HIGH);
  delay(ONESEC / 8);
  return;
}

void flashLED(int n) {
  for (int i = 0; i < n; i++)
    flashLED();
  delay(ONESEC / 2);
}

char lastChar(String s) {
  return s.charAt(s.length() - 1);
}

void myDelay(unsigned long ms) {
  unsigned long endDelay = millis() + ms;
  while (millis() < endDelay) {
    if (ts.touched()) {  // Duplicate tftAbort() in-place to minimize overhead
      unprocessed_touch = true;
      return;
    }
    delay(1);  // Seems to help with touch detection (?)
  }
  return;
}

void myPureDelay(unsigned long ms) {
  unsigned long endDelay = millis() + ms;
  while (millis() < endDelay) {
    if (ts.touched()) {  // Return ONLY - No post-return processing
      while (ts.touched())
        delay(10);
      return;
    }
    delay(1);  // Seems to help with touch detection (?)
  }
  return;
}

void displayTouchCoordinates(int x, int y) {
  uint16_t previous_X = tft.getCursorX();
  uint16_t previous_Y = tft.getCursorY();
  tft.setTextSize(1);
  tft.fillRoundRect(230, bottom_buttons_y, 100, 20, button_radius, YELLOW);
  tft.setCursor(230, bottom_buttons_y);
  tft.print("p.x = ");
  tft.println(x);
  tft.setCursor(230, bottom_buttons_y + 10);
  tft.print("p.y = ");
  tft.println(y);
  tft.setCursor(previous_X, previous_Y);
}

// Adapted from: https://afterthoughtsoftware.com/posts/convert-rgb888-to-rgb565
uint16_t uColor(uint8_t RGB[]) {

  uint16_t b = (RGB[2] >> 3) & 0x1f;
  uint16_t g = ((RGB[1] >> 2) & 0x3f) << 5;
  uint16_t r = ((RGB[0] >> 3) & 0x1f) << 11;

  return (uint16_t)(r | g | b);
}

// Special

boolean requestReboot() {
  if (digitalRead(REBOOT_BUTTON) == HIGH)
    return false;
  long startPress = millis();
  while (digitalRead(REBOOT_BUTTON) == LOW)
    ;
  if ((millis() - startPress) < REBOOT_OK)
    return false;
  return true;
}

void doReboot() {  // Software reset
  // https://github.com/TeensyUser/doc/wiki/Reset
  SCB_AIRCR = 0x05FA0004;
  asm volatile("dsb");
}

void noop(int z) {
  ;
}

// Screensaver - This is [an adaptation of] the Arduino TFT bubbles demo sketch
//               The following comment is reproduced from the example sketch

/*
	This example was adapted from ugfx http://ugfx.org
	It's a great example of many 2d objects in a 3d space (matrix transformations)
	and show[s] the capabilities of RA8875 chip.
 Tested and worked with:
 Teensy3, Teensy3.1, Arduino UNO, Arduino YUN, Arduino Leonardo, Stellaris
 Works with Arduino 1.0.6 IDE, Arduino 1.5.8 IDE, Energia 0013 IDE
*/

boolean screensaverOn() {
  if (millis() - lastTouched > SCREEN_TIMEOUT && tftMenuLevel < 2) {
    tft.fillScreen(ILI9341_BLACK);
    return true;
  }
  return false;  
}

#define NDOTS 512			// Number of dots 512
#define SCALE 4096//4096
#define INCREMENT 512//512
#define PI2 6.283185307179586476925286766559
#define RED_COLORS (32)
#define GREEN_COLORS (64)
#define BLUE_COLORS (32)

int16_t sine[SCALE+(SCALE/4)];
int16_t *cosi = &sine[SCALE/4];
int16_t angleX = 0, angleY = 0, angleZ = 0;
int16_t speedX = 0, speedY = 0, speedZ = 0;

int16_t xyz[3][NDOTS];
uint16_t col[NDOTS];
int pass = 0;

void initScreenSaver (void){
  uint16_t i;
  /* if you change the SCALE*1.25 back to SCALE, the program will
   * occassionally overrun the cosi array -- however this actually
   * produces some interesting effects as the BUBBLES LOSE CONTROL!!!!
   */
  for (i = 0; i < SCALE+(SCALE/4); i++)
    //sine[i] = (-SCALE/2) + (int)(sinf(PI2 * i / SCALE) * sinf(PI2 * i / SCALE) * SCALE);
    sine[i] = (int)(sinf(PI2 * i / SCALE) * SCALE);
}

void matrix (int16_t xyz[3][NDOTS], uint16_t col[NDOTS]){
  static uint32_t t = 0;
  int16_t x = -SCALE, y = -SCALE;
  uint16_t i, s, d;
  uint8_t red,grn,blu;

  for (i = 0; i < NDOTS; i++)
  {
    if (tftAbort())
      return;
    xyz[0][i] = x;
    xyz[1][i] = y;

    d = sqrt(x * x + y * y); 	/* originally a fastsqrt() call */
    s = sine[(t * 30) % SCALE] + SCALE;

    xyz[2][i] = sine[(d + s) % SCALE] * sine[(t * 10) % SCALE] / SCALE / 2;

    red = (cosi[xyz[2][i] + SCALE / 2] + SCALE) * (RED_COLORS - 1) / SCALE / 2;
    grn = (cosi[(xyz[2][i] + SCALE / 2 + 2 * SCALE / 3) % SCALE] + SCALE) * (GREEN_COLORS - 1) / SCALE / 2;
    blu = (cosi[(xyz[2][i] + SCALE / 2 + SCALE / 3) % SCALE] + SCALE) * (BLUE_COLORS - 1) / SCALE / 2;
    col[i] = ((red << 11) + (grn << 5) + blu);
    x += INCREMENT;
    if (x >= SCALE) x = -SCALE, y += INCREMENT;
  }
  t++;
}

void rotate (int16_t xyz[3][NDOTS], uint16_t angleX, uint16_t angleY, uint16_t angleZ){
  uint16_t i;
  int16_t tmpX, tmpY;
  int16_t sinx = sine[angleX], cosx = cosi[angleX];
  int16_t siny = sine[angleY], cosy = cosi[angleY];
  int16_t sinz = sine[angleZ], cosz = cosi[angleZ];

  for (i = 0; i < NDOTS; i++)
  {
    if (tftAbort())
      return;
    tmpX      = (xyz[0][i] * cosx - xyz[2][i] * sinx) / SCALE;
    xyz[2][i] = (xyz[0][i] * sinx + xyz[2][i] * cosx) / SCALE;
    xyz[0][i] = tmpX;
    tmpY      = (xyz[1][i] * cosy - xyz[2][i] * siny) / SCALE;
    xyz[2][i] = (xyz[1][i] * siny + xyz[2][i] * cosy) / SCALE;
    xyz[1][i] = tmpY;
    tmpX      = (xyz[0][i] * cosz - xyz[1][i] * sinz) / SCALE;
    xyz[1][i] = (xyz[0][i] * sinz + xyz[1][i] * cosz) / SCALE;
    xyz[0][i] = tmpX;
  }
}

void draw(int16_t xyz[3][NDOTS], uint16_t col[NDOTS]){
  static uint16_t oldProjX[NDOTS] = { 0 };
  static uint16_t oldProjY[NDOTS] = { 0 };
  static uint8_t oldDotSize[NDOTS] = { 0 };
  uint16_t i, projX, projY, projZ, dotSize;

  for (i = 0; i < NDOTS; i++)
  {
    if (tftAbort())
      return;
    projZ = SCALE - (xyz[2][i] + SCALE) / 4;
    projX = tft.width() / 2 + (xyz[0][i] * projZ / SCALE) / 25;
    projY = tft.height() / 2 + (xyz[1][i] * projZ / SCALE) / 25;
    dotSize = 3 - (xyz[2][i] + SCALE) * 2 / SCALE;

    tft.drawCircle (oldProjX[i], oldProjY[i], oldDotSize[i], BLACK);

    if (projX > dotSize && projY > dotSize && projX < tft.width() - dotSize && projY < tft.height() - dotSize)
    {
      tft.drawCircle (projX, projY, dotSize, col[i]);
      oldProjX[i] = projX;
      oldProjY[i] = projY;
      oldDotSize[i] = dotSize;
    }
  }
}

void oneScreenSaverIteration() 
{
  matrix(xyz, col);
  if (tftAbort())
    return;
  rotate(xyz, angleX, angleY, angleZ);
  if (tftAbort())
    return;
  draw(xyz, col);
  if (tftAbort())
    return;
  angleX += speedX;
  angleY += speedY;
  angleZ += speedZ;

  if (pass > 400) speedY = 1;
  if (pass > 800) speedX = 1;
  if (pass > 1200) speedZ = 1;
  pass++;

  if (angleX >= SCALE) {
    angleX -= SCALE;
  } 
  else if (angleX < 0) {
    angleX += SCALE;
  }

  if (angleY >= SCALE) {
    angleY -= SCALE;
  } 
  else if (angleY < 0) {
    angleY += SCALE;
  }

  if (angleZ >= SCALE) {
    angleZ -= SCALE;
  } 
  else if (angleZ < 0) {
    angleZ += SCALE;
  }
}

// The following is adapted from:
// https://codebender.cc/example/Adafruit_ILI9341/spitftbitmap#spitftbitmap.ino

/***************************************************
  This is our Bitmap drawing example for the Adafruit ILI9341 Breakout and Shield
  ----> http://www.adafruit.com/products/1651

  Check out the links above for our tutorials and wiring diagrams
  These displays use SPI to communicate, 4 or 5 pins are required to
  interface (RST is optional)
  Adafruit invests time and resources providing this open source code,
  please support Adafruit and open-source hardware by purchasing
  products from Adafruit!

  Written by Limor Fried/Ladyada for Adafruit Industries.
  MIT license, all text above must be included in any redistribution
 ****************************************************/

// This function opens a Windows Bitmap (BMP) file and
// displays it at the given coordinates.  It's sped up
// by reading many pixels worth of data at a time
// (rather than pixel by pixel).  Increasing the buffer
// size takes more of the Arduino's precious RAM but
// makes loading a little faster.  20 pixels seems a
// good balance.

#define BUFFPIXEL 1               // LM: One at a time in this adaptation

void bmpDraw(char *filename, uint8_t x, uint16_t y) {

  File     bmpFile;
  int      bmpWidth, bmpHeight;   // W+H in pixels
  uint8_t  bmpDepth;              // Bit depth (currently must be 24)
  uint32_t bmpImageoffset;        // Start of image data in file
  uint32_t rowSize;               // Not always = bmpWidth; may have padding
  uint8_t  sdbuffer[3*BUFFPIXEL]; // pixel buffer (R+G+B per pixel)
  uint8_t  buffidx = sizeof(sdbuffer); // Current position in sdbuffer
  boolean  goodBmp = false;       // Set to true on valid header parse
  boolean  flip    = true;        // BMP is stored bottom-to-top
  int      w, h, row, col;
  uint8_t  r, g, b;
  uint32_t pos = 0, startTime = millis();
  if((x >= tft.width()) || (y >= tft.height())) return;
/*
  Serial.println();
  Serial.print(F("Loading image '"));
  Serial.print(filename);
  Serial.println('\'');
*/
  // Open requested file on SD card
  if ((bmpFile = SD.open(filename)) == 0) {
//  Serial.print(F("File not found"));
    return;
  }
  // Parse BMP header
  if(read16(bmpFile) == 0x4D42) { // BMP signature
//  Serial.print(F("File size: ")); Serial.println(read32(bmpFile));
    (void)read32(bmpFile); // LM: Read and ignore file size (Remove if uncommenting preceding Serial.print)
    (void)read32(bmpFile); // Read & ignore creator bytes
    bmpImageoffset = read32(bmpFile); // Start of image data
//  Serial.print(F("Image Offset: ")); Serial.println(bmpImageoffset, DEC);
    // Read DIB header
//  Serial.print(F("Header size: ")); Serial.println(read32(bmpFile));
    (void)read32(bmpFile); // LM: Read & ignore Header size (Remove if uncommenting preceding Serial.print)
    bmpWidth  = read32(bmpFile);
    bmpHeight = read32(bmpFile);
    if(read16(bmpFile) == 1) { // # planes -- must be '1'
      bmpDepth = read16(bmpFile); // bits per pixel
//    Serial.print(F("Bit Depth: ")); Serial.println(bmpDepth);
      if((bmpDepth == 24) && (read32(bmpFile) == 0)) { // 0 = uncompressed
        goodBmp = true; // Supported BMP format -- proceed!
/*
        Serial.print(F("Image size: "));
        Serial.print(bmpWidth);
        Serial.print('x');
        Serial.println(bmpHeight);
*/
        // BMP rows are padded (if needed) to 4-byte boundary
        rowSize = (bmpWidth * 3 + 3) & ~3;
        // If bmpHeight is negative, image is in top-down order.
        // This is not canon but has been observed in the wild.
        if(bmpHeight < 0) {
          bmpHeight = -bmpHeight;
          flip      = false;
        }
        // Crop area to be loaded
        w = bmpWidth;
        h = bmpHeight;
/*
        // LM: Following does not accomodate rotation - 
        //     Make sure the BMP width and height comply before copying to SD
        if((x+w-1) >= tft.width())  w = tft.width()  - x;
        if((y+h-1) >= tft.height()) h = tft.height() - y;
*/
        // Set TFT address window to clipped image bounds
        tft.setAddrWindow(x, y, x+w-1, y+h-1);
        for (row=0; row<h; row++) { // For each scanline...
          // Seek to start of scan line.  It might seem labor-
          // intensive to be doing this on every line, but this
          // method covers a lot of gritty details like cropping
          // and scanline padding.  Also, the seek only takes
          // place if the file position actually needs to change
          // (avoids a lot of cluster math in SD library).
          if(flip) // Bitmap is stored bottom-to-top order (normal BMP)
            pos = bmpImageoffset + (bmpHeight - 1 - row) * rowSize;
          else     // Bitmap is stored top-to-bottom
            pos = bmpImageoffset + row * rowSize;
          if(bmpFile.position() != pos) { // Need seek?
            bmpFile.seek(pos);
            buffidx = sizeof(sdbuffer); // Force buffer reload
          }
          for (col=0; col<w; col++) { // For each pixel...
            // Time to read more pixel data?
            if (buffidx >= sizeof(sdbuffer)) { // Indeed
              bmpFile.read(sdbuffer, sizeof(sdbuffer));
              buffidx = 0; // Set index to beginning
            }
            // Convert pixel from BMP to TFT format, push to display
            b = sdbuffer[buffidx++];
            g = sdbuffer[buffidx++];
            r = sdbuffer[buffidx++];
//          tft.pushColor(tft.color565(r,g,b));
            // LM: Not sure what is wrong with the preceding
            //     Next is fast enough (with BUFFPIXEL = 1)
            tft.drawPixel(row, col, tft.color565(r,g,b));
          } // end pixel
        } // end scanline
        noop(startTime);            // LM: Avoid compiler warning       
/*
        Serial.print(F("Loaded in "));
        Serial.print(millis() - startTime);
        Serial.println(" ms");
*/
      } // end goodBmp
    }
  }
  bmpFile.close();
//if(!goodBmp) Serial.println(F("BMP format not recognized."));
  noop(goodBmp);                    // LM: Avoid compiler warning
}

// These read 16- and 32-bit types from the SD card file.
// BMP data is stored little-endian, Arduino is little-endian too.
// May need to reverse subscript order if porting elsewhere.

uint16_t read16(File &f) {
  uint16_t result;
  ((uint8_t *)&result)[0] = f.read(); // LSB
  ((uint8_t *)&result)[1] = f.read(); // MSB
  return result;
}

uint32_t read32(File &f) {
  uint32_t result;
  ((uint8_t *)&result)[0] = f.read(); // LSB
  ((uint8_t *)&result)[1] = f.read();
  ((uint8_t *)&result)[2] = f.read();
  ((uint8_t *)&result)[3] = f.read(); // MSB
  return result;
}

// The following comment is reproduced verbatim from the TimeTeensy3 example sketch
/*
 * TimeRTC.pde
 * example code illustrating Time library with Real Time Clock.
 * 
 */

#define TIME_HEADER  "T"        // For serial time sync message (Teensy Time Example)

void getTimeFromHost() {
  setSyncProvider(getTeensyTime);
  if (Serial.available()) {
    time_t t = processSyncMessage();
    if (t != 0) {
    Teensy3Clock.set(t);
    setTime(t);
    }
  }
}

unsigned long processSyncMessage() {
  unsigned long pctime = 0L;
  const unsigned long DEFAULT_TIME = 1357041600; // Jan 1 2013 
  if(Serial.find(TIME_HEADER)) {
     pctime = Serial.parseInt();
     return pctime;
     if( pctime < DEFAULT_TIME) { // check the value is a valid time (greater than Jan 1 2013)
       pctime = 0L; // return 0 to indicate that the time is not valid
     }
  }
  return pctime;
}

time_t getTeensyTime() {
  return Teensy3Clock.get();
}

void initClock() {
    time_t t = getTeensyTime();
    if (t != 0) {
      Teensy3Clock.set(t);
      setTime(t);
    }  
}

// LM: Generic

String sDate() {
  // Today
  String t = monthShortStr(month());
  t.concat(" ");
  t.concat(day());
  return t;  
}

String sDateY() {
  // Today
  String t = monthShortStr(month());
  t.concat(" ");
  t.concat(day());
  t.concat(", ");
  t.concat(year());
  return t;  
}

String sTime() {
  String t = "";
  if (hour() < 10)
    t.concat("0");
  t.concat(hour());
  t.concat(":");
  if (minute() < 10)
    t.concat("0");
  t.concat(minute());
  t.concat(":");
  if (second() < 10)
    t.concat("0");
  t.concat(second());
  return t;
}

String sDT() {
  String t = sDate();
  t.concat(" ");
  t.concat(sTime());
  return t;  
}

String sDTY() {
  String t = sDateY();
  t.concat(" ");
  t.concat(sTime());
  return t;  
}

// Custom time for Morse rendering
String sMorseTime() {
  String s = " - The time is ";
  String am_pm = "";  
  int h = hour();
  int m = minute();
  // Round to nearest minute
  if (second() > 30) {
    m++;
    if (m == 60) {
      m = 0;
      h++;
      if (h == 24)
        h = 0;
    }    
  }  
  if (h > 12) {
    h = h - 12;
    am_pm = "pm";
  }
  else if (h > 11)
    am_pm = "pm";
  else
    am_pm = "am";
  s.concat(String(h));
  s.concat(SPACE);
  if (m < 10)
    s.concat('0');
  s.concat(String(m));
  s.concat(SPACE);
  s.concat(am_pm);
  s.concat(".");
  return s;  
}

// Custom date for Morse rendering
String sMorseDate() {
  String s = " - Today is ";
  s.concat(DOW[weekday()-1]);
  s.concat(" ");
  s.concat(MON_NAME[month()-1]);
  s.concat(" ");
  s.concat(day());
  s.concat(", ");
  s.concat(year());
  s.concat(".");
  return s;
}

// Set time from Morse keyed input (4.1.6)
void mySetTime(String hhmm) {
  // Time only - Date is unchanged
  (void) getTeensyTime();
  int hh = hhmm.substring(0,2).toInt();
  if (hh < 0 || hh > 23) {
    outputString("? ");
    return;
  }
  int mm = hhmm.substring(2,4).toInt();
  if (mm < 0 || mm > 59) {
    outputString("? ");
    return;
  }
  tmElements_t tm;
    tm.Hour = hh;
    tm.Minute = mm;
    tm.Second = 30;
    tm.Day = day();
    tm.Month = month();
    tm.Year = year() - 1970;
  time_t t = makeTime(tm);
  Teensy3Clock.set(t);              // 4.2.3 bug fix
  setTime(t);
  outputString("R ");
  return;
}

void mySetDate(String mmddyyyy) {
  // Date only - Time is unchanged
  (void) getTeensyTime();
  int mm = mmddyyyy.substring(0,2).toInt();
  if (mm < 1 || mm > 12) {
    outputString("? ");
    return;
  }
  int dd = mmddyyyy.substring(2,4).toInt();
  if (dd < 1 || dd > 31) {          // Weak validation - not month-specific
    outputString("? ");
    return;
  }
  int yyyy = mmddyyyy.substring(4).toInt();
  if (yyyy < 2000 || yyyy > 3000) { // MOAK 4 anticipates a long run
    outputString("? ");
    return;
  }
  tmElements_t tm;
    tm.Hour = hour();
    tm.Minute = minute();
    tm.Second = second();
    tm.Day = dd;
    tm.Month = mm;
    tm.Year = yyyy - 1970;
  time_t t = makeTime(tm);
  Teensy3Clock.set(t);              // 4.2.3 bug fix
  setTime(t);
  outputString("R ");
  return;
}

String sMorseSpeed() {
  String s = " - Current Morse speed is ";
  s.concat(sndSpeed);
  s.concat(" words per minute. ");
  return s;
}


//   ----- Joseph Weizenbaum's Eliza -----
// Code and data (#include Eliza.c) based on:
// https://www.jesperjuul.net/eliza/ELIZA.BAS

#include "Data/Eliza.c"

// ELIZA.BAS symbolic names (Convenience for comparing code)
#define KEYWORD ELIZA_KEYWORDS
#define REPLIES ELIZA_REPLIES
#define N1 NUM_KEYWORDS
#define N2 NUM_CONJUGATIONS

int S[N1], R[N1], N[N1];
String WORDIN[N2/2], WORDOUT[N2/2];

const char ASTERISK = '*';          // SPACE was defined at top

void initSNR() {
  // ELIZA.BAS lines 130 - 150
  int L;
  for (int x=0; x<NUM_KEYWORDS; x++)  {
    S[x] =  RIGHT_REPLY[x+x];
    L = RIGHT_REPLY[x+x+1];
    R[x] = S[x];
    N[x] = S[x] + L - 1;
   }
  // ELIZA.BAS line 114
  for (int x=0; x<N2; x+=2) {
    WORDIN[x/2] = ELIZA_CONJUGATIONS[x];
    WORDOUT[x/2] = ELIZA_CONJUGATIONS[x+1];
  } 
}

// Eliza interface

boolean isEOT(String s) {           // True if keyed input ends with a sentence terminator (K, KN, SK etc.)
  // assume lowercase
  int L=s.length();
  if (s.charAt(L-1) != SPACE) return false;
  if (s.substring(L-3) == " k ") return true;
  String t = s.substring(L-4);
  if (t == " kn ") return true;
  if (t == " bk ") return true;
  if (t == " sk ") return true;
  return false;
}

boolean isQRT(String s) {           // True if keyed input contains a converstion terminator
  if (s.indexOf(" QRT ") >= 0)
    return true;
  if (s.indexOf(" 73 ") >= 0)
    return true;
  if (s.indexOf(" SK ") >= 0)
    return true;
  // Other terminators here - But not Eliza's 'shut up'
  return false;
}

boolean hasContent(String s) {      // Test input for non-empty content
  for (unsigned int i=0; i<s.length(); i++)
    if (s.charAt(i) != SPACE)
      return true;
    return false;
}

String sClean(String s) {           // Remove unwanted punctuation from keyed input
  while (int ndx = s.indexOf(",") >= 0)
    s = s.substring(0,ndx) + s.substring(ndx+1);
  while (int ndx = s.indexOf(".") >= 0)
    s = s.substring(0,ndx) + s.substring(ndx+1);
  return s;
}

// Eliza implementation

String sLine160() {
  //160 PRINT "HI! I'M ELIZA. WHAT'S YOUR PROBLEM?"
  String s = " - ";
  for (unsigned int i=0; i<=(random(3)); i++)
    s.concat("CQ ");
  s.concat("DE ELIZA K ");
  return s;
}

String sLine555(String sC) {        // Eliza.bas line 555
  while (sC.charAt(1) == SPACE)
    sC = sC.substring(1);
  while (int ndx = sC.indexOf("!") >= 0)
    sC = sC.substring(0,ndx) + sC.substring(ndx+1);
  return sC;
}

String sLine560(int k, String cS) {
  // If k is not c++ zero-based subscript, decrement it here
  // If R[k] is not c++ zero-based subscript, substitute -
  int j = R[k] -1;   
  String sF = REPLIES[j];           // Line 600
  R[k]++;                           // Line 610 first statement
  if (R[k] > N[k]) R[k] = S[k];     // Line 610 second statement
  if (sF.charAt(sF.length()-1) != ASTERISK)
    return sF;                      // Line 620 Last input is already saved
//if (cS != "   ")                  // Line 625 Substitute next
  if (hasContent(cS))
    return sF.substring(0, sF.length()-1) + cS;
  return "Please tell me more";     // Line 626 paraphrased
}

String sLine560(int k) {
  return sLine560(k, "");
}

String sLine400(String sIn, String sF, int L) {
  // Lines 375-410 are remarks
  // sIn (uppercase input) sF (keyword)
  // L (chaaracter position where keyword inserts in input string)
  int ndx;
  // Ignore L parameter value. Instead [re]compute it.
  L = sIn.indexOf(sF);
  String sC = sIn.substring(L+sF.length()-1);
  // To do: Cleanup possible double-space
  for (int x=0; x<N2/2; x++) {      // Line 440
    ndx = sC.indexOf(WORDIN[x]);    
    if (ndx >= 0) {
      sC = sC.substring(0, ndx) + WORDOUT[x] + sC.substring(ndx + WORDIN[x].length());
    } 
    else {       
      ndx = sC.indexOf(WORDOUT[x]);    
      if (ndx >= 0) {
        sC = sC.substring(0, ndx) + WORDIN[x] + sC.substring(ndx + WORDOUT[x].length());
      }
    }    
  }
  return sC;
}

String sLine400(String sIn, int k, int L) {
  String sF = KEYWORD[k];
  String sC = sLine400(sIn, sF, L);
  sC = sLine555(sC);                // Probably not necessary
  return sLine560(k, sC);
}  

String sEliza(String upInput) {
//String upInput;
  String sReply = "";
  String sF;                        // Alias of F$ (scratch string in BASIC program)
  int k;                            // Keyword index
  if (!hasContent(upInput))
    return (" Nothing heard BK ");    
  else if (upInput == lastInput)    // Line 255
    return " QSL - pse do not repeat BK ";
  lastInput = upInput;
  for (k=0; k<N1; k++) {            // Line 300
    if (upInput.indexOf(KEYWORD[k]) >= 0)
      break; 
  }
  if (k >= NUM_KEYWORDS)  {         // Keyword not found
    // -> Line 570 (560)
    return sLine560(k-1).concat(" BK ");    
  }
  else if (k == 12) {               // C++ index 12 = BASIC index 13
    if (upInput.indexOf(KEYWORD[28]) >= 0)
      k = 28;
    sF = KEYWORD[k];                // YOU or YOUR  
    // -> Line 390 (400)
    return sLine400(upInput, k, upInput.indexOf(sF)).concat(" BK ");
  }
  else {
    // -> Line 349
    sF = KEYWORD[k];
    // -> Line 390 (400)
    return sLine400(upInput, k, upInput.indexOf(sF)).concat(" BK ");
  }
  return sReply;                    // Not reachable
}

// MOAK-5 Additions

// Heads or Tails (based on https://www.lloydm.net/Demos/Geiger.html)

// Coin toss bitmaps
#include "Data/Penny_Heads_64.c"
#include "Data/Penny_Tails_64.c"

// Width and height of penny image
int16_t htW = 64;
int16_t htH = 60;

// Position of penny image
int16_t htX = (X_PIXELS-htW)/2;
int16_t htY = (Y_PIXELS-htH)/2;

void clearPennyImage() {
  // Paint background color where penny face was displayed
  tft.fillRect(htX, htY, htW, htH, TFT_BG);
}

void tossPenny() {
  if (millis() % 2 == 0) {
    tft.drawRGBBitmap(htX, htY, Penny_Heads_64, htW, htH);
    outputString(" - heads - ");
  }
  else {
    tft.drawRGBBitmap(htX, htY, Penny_Tails_64, htW, htH);
    outputString(" - tails - ");
  }
  clearPennyImage();
}

// Sending practice feedback (scoring and evaluation)

// Vertical bar (graph) parameters
const uint16_t VBAR_W = 10;
const uint16_t VBAR_H = 100;  
const uint16_t CUM_XPOS = X_PIXELS*3/4;
const uint16_t NON_CUM_XPOS = CUM_XPOS - 50;
const uint16_t VBAR_YPOS = Y_PIXELS/2 - VBAR_H*3/4;
const uint16_t GOOD = GREEN;
const uint16_t BAD  = RED;
const uint16_t VBAR_BORDER = BLACK;

void toggleScoring() {
  scoring = not(scoring);
  if (scoring)
    outputString(" - Scoring ON ");
  else
    outputString(" - Scoring OFF ");
}

void clearVbar() {
  tft.fillRect(CUM_XPOS, VBAR_YPOS, VBAR_W, VBAR_H, TFT_BG);
}

void displayScoreAsText(uint16_t xPos, uint16_t yPos, uint16_t score) {
  tft.setCursor(xPos, yPos);
  tft.setTextColor(TFT_TXT);
  tft.setTextSize(1);
  tft.println(String(score) + "%");
}

void displayVbarLabel(uint16_t xPos, uint16_t yPos, String(label)) {
  tft.setCursor(xPos, yPos);
  tft.setTextColor(TFT_TXT);
  tft.setTextSize(1);
  tft.println(label);
}

void displayNonCumVbar(int num, int den) {
  if (den > 0) {
    uint16_t score = num * 100 / den;
    uint16_t scoreLengthCorrection = -2;
    if (score == 100)
      scoreLengthCorrection = -5;
    displayScoreAsText(NON_CUM_XPOS+scoreLengthCorrection, VBAR_YPOS-10, score);
    score = 100 - score;            // Invert for graph
    tft.fillRect(NON_CUM_XPOS, VBAR_YPOS, VBAR_W, score, BAD);
    tft.fillRect(NON_CUM_XPOS, VBAR_YPOS+score+1, VBAR_W, VBAR_H-score-1, GOOD);
    tft.drawRect(NON_CUM_XPOS, VBAR_YPOS, VBAR_W, VBAR_H, VBAR_BORDER);
  }
}

void displayCumVbar(int num, int den) {
  if (den > 0) {
    uint16_t score = num * 100 / den;
    uint16_t scoreLengthCorrection = -2;
    if (score == 100)
      scoreLengthCorrection = -5;
    displayScoreAsText(CUM_XPOS+scoreLengthCorrection, VBAR_YPOS-10, score);
    score = 100 - score;            // Invert for graph
    tft.fillRect(CUM_XPOS, VBAR_YPOS, VBAR_W, score, BAD);
    tft.fillRect(CUM_XPOS, VBAR_YPOS+score+1, VBAR_W, VBAR_H-score-1, GOOD);
    tft.drawRect(CUM_XPOS, VBAR_YPOS, VBAR_W, VBAR_H, VBAR_BORDER);
    displayVbarLabel(CUM_XPOS-15, VBAR_YPOS+VBAR_H+5, "Overall");
  }
}

// Evaluate touch calibration accuracy
void calibrateTFT() {               // [Re]calibrate TFT (Not implemented in V.4.2)
  const int N_CALS = 20;
  uint16_t xData[N_CALS];
  uint16_t yData[N_CALS];
  uint16_t testX;
  uint16_t testY;
  tft.fillRect(0, 0, X_PIXELS, Y_PIXELS, TFT_BG);
  tft.drawRect(1, 1, X_PIXELS-2, Y_PIXELS-2, BLUE);
  tftPrintText_NoClear(" Carefully touch each dot", 0, Y_PIXELS/2);
  delay(TWOSEC);                    // Do not interrupt
  tftPrintText_NoClear(" Carefully touch each dot", 0, Y_PIXELS/2, TFT_BG);
  xData[0]=0; noop(xData[0]);       // Debug - Avoid compiler warning until array is used
  yData[0]=0; noop(yData[0]);
  for (int i=0; i<N_CALS; i++) {    // Data acquisition loop
    testX = i%5 * (X_PIXELS/5) + (X_PIXELS/10);
    testY = (i/5)%4 * (Y_PIXELS/4) + (Y_PIXELS/8);
    drawCalibrationPoint(testX, testY, RED);
    Serial.print("Test point ");    // Debug
    Serial.print(i+1);
    Serial.print(": (");
    Serial.print(testX);
    Serial.print(", ");
    Serial.print(testY);
    Serial.println(")");
    // Coding in progress here
    delay(HALFSEC);                 // Placeholder for touch processing
    drawCalibrationPoint(testX, testY, GREEN);
    delay(HALFSEC);
    drawCalibrationPoint(testX, testY, TFT_BG);
  }
}

void drawCalibrationPoint(uint16_t x, uint16_t y, uint16_t color) {
  for (int i=x-2; i<=(x+2); i++)
    for (int j=y-2; j<=(y+2); j++)
      tft.drawPixel(i, j, color);
}