/*
 * Lloyd Milligan (WA4EFS) - February 2024
 * © CC attribution - https://creativecommons.org/licenses/by/3.0/us/
 *
 * Uses code examples from: https://www.youtube.com/watch?v=ytJTSyUYifY
 *                     and: https://www.youtube.com/watch?v=yA6KrFErj6I
 *
 * Click nbfs://nbhost/SystemFileSystem/Templates/GUIForms/JFrame.java to edit this template
 *
 * April 2024 revision - Add rigctl interface (localhost and default port)
 *
 * This application includes working examples of -
 *      Bidirectional windows serial communication (requires com.fazecast.jSerialComm)
 *          Download from: https://fazecast.github.io/jSerialComm/
 *      TCP/IP client communication
 *      Text file buffered writer and print writer facilities
 *      Swing Timer generated events
 *      Windows local date/time related processing
 *
 */

package my.QMX;

import com.fazecast.jSerialComm.*;
import javax.swing.JOptionPane;
import static javax.swing.JOptionPane.ERROR_MESSAGE;
import java.io.OutputStream;
import java.io.IOException;
import javax.swing.Timer;
// Log support
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.time.ZoneOffset;
// Rig control
import java.net.*;
import java.io.*;

// Debug
import java.util.Random;
import java.io.PrintWriter;

public class qmxUI extends javax.swing.JFrame {
    /**
     * Creates new form qmxUI
     */
    static final boolean DEBUG_MODE = false;
    static final int FRAME_WIDTH_DELTA = -95;
    
    SerialPort qmxPort;
    OutputStream txStream;
    
    // Command line parameters
    static boolean enableLog = false;
    static boolean verboseLogTuning = false;
    static boolean contactLog = false;

    // Includes unreferenced declarations (deprecated or future use)
    // General Appearance
    static final int BOTTOM_ROW_VDELTA = -5;
    static final int DEBUG_AREA_VDELTA = -10;
    static final int BELOW_BAND_ROW_VDELTA = 30;
    static final int LOG_TEXT_WIDTH = 140;
    // QMX-specific serial interface values:
    static final int BAUD = 9600;
    int DATABITS = 8;
    int STOPBITS = 1;
    int NO_PARITY = 0;
    // VFO display parameters
    static final int VCOLS = 10;
    static final int VFOA_HPOS = 50;
    static final int VFOA_VPOS = 15;
    static final int VFOB_HPOS = 50;
    static final int VFOB_VPOS = 65;
    javax.swing.JTextField[] vfoA = new javax.swing.JTextField[VCOLS];
    javax.swing.JTextField[] vfoB = new javax.swing.JTextField[VCOLS];
    static final String DLM = ".";
    String lastVFOA = "00000000000";    // Pace updating
    String lastVFOB = "00000000000";
    // Ham radio bands (10's of Hz 'dHz')
    // Insert lower bands here
    static final int LB20 = 14000;
    static final int UB20 = 14350;
    static final int LB17 = 18068;
    static final int UB17 = 18168;
    static final int LB15 = 21000;
    static final int UB15 = 21450;
    static final int LB12 = 24890;
    static final int UB12 = 24990;
/*
    // 11 meter band  - Not US - Pseudo bounds (not used)
    static final int LB11 = 27000;
    static final int UB11 = 28000;
*/
    static final int LB10 = 28000;
    static final int UB10 = 29700;
    static final int BBTN_HPOS = 50;
    static final int BBTN_HDELTA = 15;
    static final int BBTN_VPOS = 115;
    // Band change-to frequencies
    // VFO-A
    String bS20A = "00014074000";
    String bS17A = "00018104600";
    String bS15A = "00021094600";
    String bS12A = "00024924600";
    String bS11A = "00027245000";    // Not used
    String bS10A = "00028124600";
    // VFO-B
    String bS20B = "00014074000";
    String bS17B = "00018104600";
    String bS15B = "00021094600";
    String bS12B = "00024924600";
    String bS11B = "00027245000";    // Not used
    String bS10B = "00028124600";
    // Tuning
    int rxVFO = 0;          // 0 = VFOA, 1 = VFOB
    int txVFO = 0;          // Ditto
    boolean splitVFO = false;
    int targetVFO = 0;      // 0=VFOA, 1=VFOB
    int slowHz = 10;        // Slow tuning rate
    int fastHz = 500;       // Fast tuning rate
    java.awt.Color tuneableVFOlabelColor = new java.awt.Color(255, 255, 128);
    // Band changing
    static final int NBANDS = 6;
    int [] bandDefaultFrequency = {14074, 18104, 21094, 24924, 27245, 28124};
    int currentBandNdx = -1;
    int previousBandNdx = -1;
    // CAT
    static final int CAT_UPDATE_INTERVAL = 120; // milliseconds
    static final char EOC = ';';
    static final char UNK = '?';
    String sCommand = "";
    String sendOnce = "";
    // Next is sent on every timer update interval event
    String sendAlways = "FA" + EOC + "MD" + EOC + "TQ" + EOC;
    int catMsgNdx = 0;
    int catMsgCount = 4;
    String [] catMsgList = {"FA" + EOC,
                            "FB" + EOC,
                            "MD" + EOC + "TQ" + EOC,
                            "FR" + EOC + "FT" + EOC + "SP" + EOC};
    boolean ritStatus = false;
    String sRIT = "";
    char cOperMode;
    int xcvrStatus = 0;
    int operMode = 0;
    int rcvVFO = 0;
    boolean simplex = true;
    boolean TxMode = false;     // True iff transmit is on
    // Other visual
    static final int RLBLS_HDELTA = 12;
    static final int BANDLBL_VDELTA = 4;
    static final int MODELBL_VDELTA = BANDLBL_VDELTA + 20;
    // Log file
    boolean logActive = false;
    static final char TAB = 9;
    static final char SP = 32;
    String logFilePath = "./";
    String logFileName = "qmxLog.txt";
    FileWriter logFileWriter;
    BufferedWriter logFileBufferedWriter;
    boolean lastLoggedSplitVFO = false;
    boolean lastLoggedTxMode = false;
    LocalDateTime logDT;                // Log date/time stamp
    LocalDateTime lastLogDT;            // For pacing certain log entries
    LocalDateTime lastRxTxTransitionDT; // Ditto
    LocalDateTime lastLoggedFreqDT;
    private final int TxRxTransitionDuration = 5;   // Seconds
    private final int MIN_VFO_LOG_INTERVAL = 1;     // Ditto - Pace log entries for VFO tuning
    char lastOperMode = '0';
    int lastRxVFO = -1;
     // Rig control
    boolean rigctlEnabled = false;
    String rigctlHost = "localhost";
    int rigctlPort = 4532; // Hamlib rigctl
    OutputStream rcOutput;
    PrintWriter rcWriter;
    // static final char EOL = 13;
    InputStream rcInput;
    BufferedReader rcReader;
    int rigctlVFO = -1;
    int freqCorrection = 0;
    int rigctlMod = 5;     // Pace Rig control frequency checks
    private final int LOG_BTN_TO_RIGCTL_BTN_Y_DELTA = 27;
    // Debug
    boolean debugFileEnabled = true;
    String debugFilePath = "./";
    String debugFileName = "debugLog.txt";
    public static FileWriter debugFileWriter;
    PrintWriter debugPrint;
    boolean debugOnce = true;
    private final int MAX_DEBUG_PRINTS = 5;
    int debugPrintCount = 0;

    public qmxUI() {
        setLocationRelativeTo(null);
        initComponents();

        // Next is a horrible kludge - Runtime width differs from design width!
        this.setBounds(this.getX(), this.getY(), getWidth() + FRAME_WIDTH_DELTA, this.getHeight());
        
        final boolean debugStatus = false;            // Enable/disable debugData text area as needed for testing
        debugData.setEnabled(debugStatus);
        debugData.setEditable(debugStatus);
        debugData.setVisible(debugStatus);
        //debugData.setText(Integer.toString(lcdDisplayPanel.getX()));
        
        
        prototypeTextField.setVisible(false);
        prototypeUpDnBtn.setVisible(false);
        connectBtn.setEnabled(true);
        disconnectBtn.setEnabled(false);
        mainInterfacePanel.setLayout(null);
        
        // Frequency display components
        for (int i=0; i<VCOLS; i++) {
            vfoA[i] = new javax.swing.JTextField("");
            vfoB[i] = new javax.swing.JTextField("");
            mainInterfacePanel.add(vfoA[i]);
            mainInterfacePanel.add(vfoB[i]);
            vfoA[i].setBackground(prototypeTextField.getBackground());
            vfoB[i].setBackground(prototypeTextField.getBackground());
            vfoA[i].setForeground(prototypeTextField.getForeground());
            vfoB[i].setForeground(prototypeTextField.getForeground());
            vfoA[i].setSize(prototypeTextField.getSize());
            vfoB[i].setSize(prototypeTextField.getSize());
            vfoA[i].setLocation(i*prototypeTextField.getWidth()+VFOA_HPOS,VFOA_VPOS);
            vfoB[i].setLocation(i*prototypeTextField.getWidth()+VFOB_HPOS,VFOB_VPOS);
            vfoA[i].enableInputMethods(true);
            vfoB[i].enableInputMethods(true);
            vfoA[i].setText("");
            vfoB[i].setText("");
            vfoA[i].setFont(prototypeTextField.getFont());
            vfoB[i].setFont(prototypeTextField.getFont());
            vfoA[i].setEditable(false);
            vfoB[i].setEditable(false);
            vfoA[i].setVisible(true);
            vfoB[i].setVisible(true);
        }
        disableVFOA();
        disableVFOB();

        // QMX Mode
        modeLbl.setText("");
        modeLbl.setVisible(false);
        modeLbl.setLocation(VFOA_HPOS + VCOLS*prototypeTextField.getWidth() + RLBLS_HDELTA, VFOA_VPOS + MODELBL_VDELTA);

        // Band
        bandLbl.setLocation(VFOA_HPOS + VCOLS*prototypeTextField.getWidth() + RLBLS_HDELTA, VFOA_VPOS + BANDLBL_VDELTA);
        bandLbl.setText("");
        bandLbl.setVisible(false);
        rBtn20.setLocation(BBTN_HPOS, BBTN_VPOS);
        rBtn17.setLocation(BBTN_HPOS + rBtn20.getWidth() + BBTN_HDELTA, BBTN_VPOS);
        rBtn15.setLocation(BBTN_HPOS + 2*(rBtn20.getWidth() + BBTN_HDELTA), BBTN_VPOS);
        rBtn12.setLocation(BBTN_HPOS + 3*(rBtn20.getWidth() + BBTN_HDELTA), BBTN_VPOS);
        rBtn10.setLocation(BBTN_HPOS + 4*(rBtn20.getWidth() + BBTN_HDELTA), BBTN_VPOS);
        hideBandButtons();

        // Tuning (To do: Parameterize constants)
        upBtn.setLocation(VFOB_HPOS+338, VFOB_VPOS - 5);
        dnBtn.setLocation(VFOB_HPOS+338, VFOB_VPOS + 20);
        tuningRateCheckBox.setLocation(VFOB_HPOS+330, VFOB_VPOS+50);
        vfoALbl.setLocation(VFOA_HPOS - 15, VFOA_VPOS + 2);
        vfoBLbl.setLocation(VFOA_HPOS - 15, VFOB_VPOS + 2);
        hideTuning();
        
        // Tx/Rx
        TxIndicatorTextField.setVisible(false);
        TxIndicatorTextField.setLocation(VFOA_HPOS - 45, VFOA_VPOS);
        TxIndicatorTextField.setBackground(new java.awt.Color(0,153,0));  // 'R' Green
        splitLbl.setLocation(VFOA_HPOS - 45, VFOA_VPOS + 28);
        //splitLbl.setSize(40,16);
        hideSplit();
        
        // Set QMX time button (Row below band selection components)
        setTimeBtn.setLocation(rBtn20.getX(), rBtn20.getY() + BELOW_BAND_ROW_VDELTA);
        setTimeBtn.setToolTipText("Copy computer time to QMX");
        setTimeBtn.setEnabled(false);
        setTimeBtn.setVisible(false);
        jLogTextField.setLocation(setTimeBtn.getX() + setTimeBtn.getWidth(), rBtn20.getY() + BELOW_BAND_ROW_VDELTA);
        jLogTextField.setSize(LOG_TEXT_WIDTH,jLogTextField.getHeight());
        jLogTextField.setVisible(false);
        jLogTextField.setToolTipText("Enter text to log");
        logBtn.setLocation(jLogTextField.getX() + jLogTextField.getWidth(), rBtn20.getY() + BELOW_BAND_ROW_VDELTA);
        logBtn.setToolTipText("Add text to log");
        logBtn.setEnabled(false);
        logBtn.setVisible(false);
        
        // Rig control
	sdr2qmxBtn.setLocation(logBtn.getX(),logBtn.getY() + LOG_BTN_TO_RIGCTL_BTN_Y_DELTA);
        sdr2qmxBtn.setToolTipText("Tune QMX to SDR frequency");
        sdr2qmxBtn.setEnabled(false);
        sdr2qmxBtn.setVisible(false);
        
        // General
        comSelectComboBox.requestFocusInWindow();
        testButton.setText("SDR++");      // Repurpose test button
        testButton.setEnabled(true);        // Toggle as needed - Test rigctl
        testButton.setText("SDR++");
        // Next corresponds to an arbitrary past time (March 5, 2024)
        lastLoggedFreqDT = LocalDateTime.ofEpochSecond(1709596800, 0, ZoneOffset.UTC);
        
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        BandBtnGroup = new javax.swing.ButtonGroup();
        jPanel1 = new javax.swing.JPanel();
        comSelectComboBox = new javax.swing.JComboBox<>();
        disconnectBtn = new javax.swing.JButton();
        connectBtn = new javax.swing.JButton();
        comPortLbl = new javax.swing.JLabel();
        testButton = new javax.swing.JButton();
        jScrollPane1 = new javax.swing.JScrollPane();
        debugData = new javax.swing.JTextArea();
        prototypeTextField = new javax.swing.JTextField();
        prototypeUpDnBtn = new javax.swing.JButton();
        mainInterfacePanel = new javax.swing.JPanel();
        vfoALbl = new javax.swing.JLabel();
        modeLbl = new javax.swing.JLabel();
        bandLbl = new javax.swing.JLabel();
        upBtn = new javax.swing.JButton();
        dnBtn = new javax.swing.JButton();
        tuningRateCheckBox = new javax.swing.JCheckBox();
        TxIndicatorTextField = new javax.swing.JTextField();
        rBtn20 = new javax.swing.JRadioButton();
        rBtn17 = new javax.swing.JRadioButton();
        rBtn15 = new javax.swing.JRadioButton();
        rBtn12 = new javax.swing.JRadioButton();
        rBtn10 = new javax.swing.JRadioButton();
        vfoBLbl = new javax.swing.JLabel();
        splitLbl = new javax.swing.JLabel();
        setTimeBtn = new javax.swing.JButton();
        jLogTextField = new javax.swing.JTextField();
        logBtn = new javax.swing.JButton();
        sdr2qmxBtn = new javax.swing.JButton();

        setDefaultCloseOperation(javax.swing.WindowConstants.EXIT_ON_CLOSE);
        setTitle("QMX++");

        jPanel1.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));
        jPanel1.setFont(new java.awt.Font("Courier New", 0, 24)); // NOI18N

        comSelectComboBox.setName(""); // NOI18N
        comSelectComboBox.addPopupMenuListener(new javax.swing.event.PopupMenuListener() {
            public void popupMenuCanceled(javax.swing.event.PopupMenuEvent evt) {
            }
            public void popupMenuWillBecomeInvisible(javax.swing.event.PopupMenuEvent evt) {
            }
            public void popupMenuWillBecomeVisible(javax.swing.event.PopupMenuEvent evt) {
                comSelectComboBoxPopupMenuWillBecomeVisible(evt);
            }
        });

        disconnectBtn.setText("Disconnect");
        disconnectBtn.setName(""); // NOI18N
        disconnectBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                disconnectBtnActionPerformed(evt);
            }
        });

        connectBtn.setText("Connect");
        connectBtn.setName(""); // NOI18N
        connectBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                connectBtnActionPerformed(evt);
            }
        });

        comPortLbl.setText("COM Port:");

        testButton.setText("Test");
        testButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                testButtonActionPerformed(evt);
            }
        });

        debugData.setEditable(false);
        debugData.setColumns(20);
        debugData.setFont(new java.awt.Font("Lucida Console", 0, 12)); // NOI18N
        debugData.setRows(5);
        jScrollPane1.setViewportView(debugData);

        prototypeTextField.setBackground(new java.awt.Color(0, 102, 0));
        prototypeTextField.setFont(new java.awt.Font("Courier New", 0, 36)); // NOI18N
        prototypeTextField.setForeground(new java.awt.Color(255, 255, 102));
        prototypeTextField.setText("0");

        prototypeUpDnBtn.setBackground(new java.awt.Color(102, 0, 0));
        prototypeUpDnBtn.setEnabled(false);

        javax.swing.GroupLayout jPanel1Layout = new javax.swing.GroupLayout(jPanel1);
        jPanel1.setLayout(jPanel1Layout);
        jPanel1Layout.setHorizontalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, jPanel1Layout.createSequentialGroup()
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addGap(18, 18, 18)
                        .addComponent(comPortLbl)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                        .addComponent(comSelectComboBox, javax.swing.GroupLayout.PREFERRED_SIZE, 75, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(18, 18, 18)
                        .addComponent(disconnectBtn)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(connectBtn)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                        .addComponent(testButton))
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                        .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(jScrollPane1, javax.swing.GroupLayout.PREFERRED_SIZE, 410, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addGroup(jPanel1Layout.createSequentialGroup()
                                .addGap(16, 16, 16)
                                .addComponent(prototypeTextField, javax.swing.GroupLayout.PREFERRED_SIZE, 33, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addGap(18, 18, 18)
                                .addComponent(prototypeUpDnBtn, javax.swing.GroupLayout.PREFERRED_SIZE, 33, javax.swing.GroupLayout.PREFERRED_SIZE)))))
                .addGap(14, 14, 14))
        );
        jPanel1Layout.setVerticalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel1Layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(prototypeTextField, javax.swing.GroupLayout.PREFERRED_SIZE, 45, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(prototypeUpDnBtn, javax.swing.GroupLayout.PREFERRED_SIZE, 5, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(18, 18, 18)
                .addComponent(jScrollPane1, javax.swing.GroupLayout.PREFERRED_SIZE, 30, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(disconnectBtn)
                    .addComponent(connectBtn)
                    .addComponent(testButton)
                    .addComponent(comSelectComboBox, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(comPortLbl))
                .addGap(18, 18, 18))
        );

        mainInterfacePanel.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));

        vfoALbl.setBackground(new java.awt.Color(255, 255, 255));
        vfoALbl.setFont(new java.awt.Font("Courier New", 0, 18)); // NOI18N
        vfoALbl.setForeground(new java.awt.Color(0, 51, 153));
        vfoALbl.setText("A");
        vfoALbl.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));
        vfoALbl.setOpaque(true);
        vfoALbl.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mousePressed(java.awt.event.MouseEvent evt) {
                vfoALblMousePressed(evt);
            }
        });

        modeLbl.setBackground(new java.awt.Color(153, 255, 255));
        modeLbl.setFont(new java.awt.Font("Courier New", 1, 12)); // NOI18N
        modeLbl.setForeground(new java.awt.Color(0, 51, 153));
        modeLbl.setText("digi");
        modeLbl.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));

        bandLbl.setBackground(new java.awt.Color(153, 255, 255));
        bandLbl.setFont(new java.awt.Font("Courier New", 1, 12)); // NOI18N
        bandLbl.setForeground(new java.awt.Color(0, 51, 153));
        bandLbl.setText("band");
        bandLbl.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));

        upBtn.setText("▲");
        upBtn.setHorizontalTextPosition(javax.swing.SwingConstants.CENTER);
        upBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                upBtnActionPerformed(evt);
            }
        });

        dnBtn.setText("▼");
        dnBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                dnBtnActionPerformed(evt);
            }
        });

        tuningRateCheckBox.setText("Fast");

        TxIndicatorTextField.setEditable(false);
        TxIndicatorTextField.setBackground(new java.awt.Color(153, 0, 0));
        TxIndicatorTextField.setFont(new java.awt.Font("Courier New", 0, 18)); // NOI18N
        TxIndicatorTextField.setForeground(new java.awt.Color(255, 255, 204));
        TxIndicatorTextField.setHorizontalAlignment(javax.swing.JTextField.CENTER);
        TxIndicatorTextField.setText("R");

        BandBtnGroup.add(rBtn20);
        rBtn20.setText("20");
        rBtn20.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                rBtn20ActionPerformed(evt);
            }
        });

        BandBtnGroup.add(rBtn17);
        rBtn17.setText("17");
        rBtn17.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                rBtn17ActionPerformed(evt);
            }
        });

        BandBtnGroup.add(rBtn15);
        rBtn15.setText("15");
        rBtn15.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                rBtn15ActionPerformed(evt);
            }
        });

        BandBtnGroup.add(rBtn12);
        rBtn12.setText("12");
        rBtn12.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                rBtn12ActionPerformed(evt);
            }
        });

        BandBtnGroup.add(rBtn10);
        rBtn10.setText("10");
        rBtn10.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                rBtn10ActionPerformed(evt);
            }
        });

        vfoBLbl.setBackground(new java.awt.Color(255, 255, 255));
        vfoBLbl.setFont(new java.awt.Font("Courier New", 0, 18)); // NOI18N
        vfoBLbl.setForeground(new java.awt.Color(0, 51, 153));
        vfoBLbl.setText("B");
        vfoBLbl.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));
        vfoBLbl.setOpaque(true);
        vfoBLbl.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mousePressed(java.awt.event.MouseEvent evt) {
                vfoBLblMousePressed(evt);
            }
        });

        splitLbl.setFont(new java.awt.Font("Segoe UI", 0, 11)); // NOI18N
        splitLbl.setText("Split Off");

        setTimeBtn.setText("Set QMX Time");
        setTimeBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                setTimeBtnActionPerformed(evt);
            }
        });

        logBtn.setText("◄ Log");
        logBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                logBtnActionPerformed(evt);
            }
        });

        sdr2qmxBtn.setText("SDR→QMX");
        sdr2qmxBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                sdr2qmxBtnActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout mainInterfacePanelLayout = new javax.swing.GroupLayout(mainInterfacePanel);
        mainInterfacePanel.setLayout(mainInterfacePanelLayout);
        mainInterfacePanelLayout.setHorizontalGroup(
            mainInterfacePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(mainInterfacePanelLayout.createSequentialGroup()
                .addGroup(mainInterfacePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(mainInterfacePanelLayout.createSequentialGroup()
                        .addContainerGap()
                        .addComponent(splitLbl)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                        .addComponent(bandLbl, javax.swing.GroupLayout.PREFERRED_SIZE, 32, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addGroup(mainInterfacePanelLayout.createSequentialGroup()
                        .addGap(17, 17, 17)
                        .addGroup(mainInterfacePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                            .addComponent(vfoALbl, javax.swing.GroupLayout.PREFERRED_SIZE, 14, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(TxIndicatorTextField, javax.swing.GroupLayout.PREFERRED_SIZE, 25, javax.swing.GroupLayout.PREFERRED_SIZE))
                        .addGap(29, 29, 29)
                        .addGroup(mainInterfacePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addGroup(mainInterfacePanelLayout.createSequentialGroup()
                                .addComponent(rBtn20)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(rBtn17)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(rBtn15)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(rBtn12)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(rBtn10)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, 33, Short.MAX_VALUE)
                                .addComponent(tuningRateCheckBox)
                                .addGap(18, 18, 18)
                                .addComponent(upBtn)
                                .addGap(8, 8, 8)
                                .addComponent(dnBtn)
                                .addGap(18, 18, 18)
                                .addComponent(modeLbl))
                            .addGroup(mainInterfacePanelLayout.createSequentialGroup()
                                .addGroup(mainInterfacePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                    .addComponent(vfoBLbl, javax.swing.GroupLayout.PREFERRED_SIZE, 14, javax.swing.GroupLayout.PREFERRED_SIZE)
                                    .addGroup(mainInterfacePanelLayout.createSequentialGroup()
                                        .addComponent(setTimeBtn)
                                        .addGap(18, 18, 18)
                                        .addComponent(jLogTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                                        .addComponent(logBtn)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                                        .addComponent(sdr2qmxBtn)))
                                .addGap(0, 0, Short.MAX_VALUE)))))
                .addGap(15, 15, 15))
        );
        mainInterfacePanelLayout.setVerticalGroup(
            mainInterfacePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(mainInterfacePanelLayout.createSequentialGroup()
                .addGroup(mainInterfacePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(mainInterfacePanelLayout.createSequentialGroup()
                        .addGap(14, 14, 14)
                        .addGroup(mainInterfacePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                            .addComponent(vfoALbl, javax.swing.GroupLayout.PREFERRED_SIZE, 18, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(vfoBLbl, javax.swing.GroupLayout.PREFERRED_SIZE, 18, javax.swing.GroupLayout.PREFERRED_SIZE))
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                        .addComponent(splitLbl))
                    .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, mainInterfacePanelLayout.createSequentialGroup()
                        .addContainerGap()
                        .addComponent(bandLbl, javax.swing.GroupLayout.PREFERRED_SIZE, 14, javax.swing.GroupLayout.PREFERRED_SIZE)))
                .addGroup(mainInterfacePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(mainInterfacePanelLayout.createSequentialGroup()
                        .addGap(18, 18, 18)
                        .addGroup(mainInterfacePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                            .addComponent(upBtn)
                            .addComponent(dnBtn)
                            .addComponent(tuningRateCheckBox)
                            .addComponent(TxIndicatorTextField, javax.swing.GroupLayout.PREFERRED_SIZE, 22, javax.swing.GroupLayout.PREFERRED_SIZE)))
                    .addGroup(mainInterfacePanelLayout.createSequentialGroup()
                        .addGap(5, 5, 5)
                        .addGroup(mainInterfacePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addGroup(mainInterfacePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                                .addComponent(rBtn12)
                                .addComponent(rBtn10)
                                .addComponent(rBtn15)
                                .addComponent(rBtn17)
                                .addComponent(rBtn20))
                            .addComponent(modeLbl, javax.swing.GroupLayout.PREFERRED_SIZE, 14, javax.swing.GroupLayout.PREFERRED_SIZE))))
                .addGap(18, 18, 18)
                .addGroup(mainInterfacePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(setTimeBtn)
                    .addComponent(jLogTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(logBtn)
                    .addComponent(sdr2qmxBtn))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addGap(23, 23, 23)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                    .addComponent(mainInterfacePanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(jPanel1, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                .addContainerGap(25, Short.MAX_VALUE))
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(mainInterfacePanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addGap(18, 18, 18)
                .addComponent(jPanel1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addGap(25, 25, 25))
        );

        jPanel1.getAccessibleContext().setAccessibleName("");
        mainInterfacePanel.getAccessibleContext().setAccessibleName("");
        mainInterfacePanel.getAccessibleContext().setAccessibleDescription("");

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void comSelectComboBoxPopupMenuWillBecomeVisible(javax.swing.event.PopupMenuEvent evt) {//GEN-FIRST:event_comSelectComboBoxPopupMenuWillBecomeVisible
        // TODO add your handling code here:
        comSelectComboBox.removeAllItems();
        SerialPort [] portLists = SerialPort.getCommPorts();
        for (SerialPort port : portLists) {
            comSelectComboBox.addItem(port.getSystemPortName());
        }
    }//GEN-LAST:event_comSelectComboBoxPopupMenuWillBecomeVisible

    private void connectBtnActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_connectBtnActionPerformed
        // TODO add your handling code here:
        // Debug
        if (debugFileEnabled) {
            try {
                debugFileWriter = new FileWriter(debugFilePath+debugFileName, true);
                debugPrint = new PrintWriter(debugFileWriter);
            }
            catch (IOException e) {
                debugFileEnabled = false;
            }
        }
        
        try {
            SerialPort [] portLists = SerialPort.getCommPorts();
            qmxPort = portLists[comSelectComboBox.getSelectedIndex()];
            qmxPort.setBaudRate(BAUD);
            qmxPort.setNumDataBits(DATABITS);
            qmxPort.setNumStopBits(STOPBITS);
            qmxPort.setParity(NO_PARITY);
            qmxPort.openPort();
            
            if (qmxPort.isOpen()) {
                // JOptionPane.showMessageDialog(this,qmxPort.getDescriptivePortName() + " connected!");
                // Log file
                if (enableLog) {
                    try {
                        logFileWriter = new FileWriter(logFilePath+logFileName, true);
                        logFileBufferedWriter = new BufferedWriter(logFileWriter);
                        logActive = true;
                    }
                    catch (IOException e) {
                        JOptionPane.showMessageDialog(this,"Cannot open debug file!");
                        logActive = false;
                    }
                }
                logActivity("Connected to QMX");
                connectBtn.setEnabled(false);
                comSelectComboBox.setEnabled(false);
                disconnectBtn.setEnabled(true);
                enableVFOA();
                enableVFOB();
                modeLbl.setVisible(true);
                bandLbl.setVisible(true);
                showBandButtons();
                // Get transceiver information  here
                CAT_update_timer.start();
                Serial_EventBasedRead(qmxPort);
                testButton.enableInputMethods(true);
                TxIndicatorTextField.setVisible(true);
                // Re-init to force VFO update after disconnect/reconnect cycle
                lastVFOA = "00000000000";
                lastVFOB = "00000000000";
                catMsgNdx = 0;
                selectVFOA();   // Default
                setTimeBtn.setEnabled(true);
                setTimeBtn.setVisible(true);
                jLogTextField.setVisible(true);
                logBtn.setEnabled(true);
                logBtn.setVisible(true);
           }
            else {
                JOptionPane.showMessageDialog(this,qmxPort.getDescriptivePortName() + " connect failed!");
            }
        }
        catch(ArrayIndexOutOfBoundsException a) {
            JOptionPane.showMessageDialog(this,"Please choose valid COM port!", "ERROR", ERROR_MESSAGE);
        }
        catch(Exception b) {
            JOptionPane.showMessageDialog(this,b, "ERROR", ERROR_MESSAGE);
        }
    }//GEN-LAST:event_connectBtnActionPerformed

    private void disconnectBtnActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_disconnectBtnActionPerformed
        // TODO add your handling code here:
        if (qmxPort.isOpen()) {
            qmxPort.closePort();
            connectBtn.setEnabled(true);
            disconnectBtn.setEnabled(false);
            comSelectComboBox.setEnabled(true);
            debugData.setText("");
            disableVFOA();
            disableVFOB();
            modeLbl.setVisible(false);
            bandLbl.setVisible(false);
            CAT_update_timer.stop();
            hideTuning();
            hideBandButtons();
            testButton.enableInputMethods(false);
            TxIndicatorTextField.setVisible(false);
            logActivity("Disconnected from QMX");
            setTimeBtn.setEnabled(false);
            setTimeBtn.setVisible(false);
            jLogTextField.setVisible(false);
            logBtn.setEnabled(false);
            logBtn.setVisible(false);
            if (logActive) {
                try {
                    logFileBufferedWriter.close();
                    logActive = false;
                }
                catch (IOException e) {
                    logActive = false;
                }
            }
       }
        if (debugFileEnabled) { 
            debugPrint.close();
       }
    }//GEN-LAST:event_disconnectBtnActionPerformed

    private void testButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_testButtonActionPerformed
        // TODO add your handling code here:
        
        // The test button has been repurposed. It is not a rig control toggle.
        toggleRigctlEnabled();
        
    }//GEN-LAST:event_testButtonActionPerformed

    private void upBtnActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_upBtnActionPerformed
        // TODO add your handling code here:
        String s = lastVFOA;
        if (targetVFO == 1)
            s = lastVFOB;
        int iFreq = Integer.parseInt(s);
        if (tuningRateCheckBox.isSelected())
            iFreq += fastHz;
        else
            iFreq += slowHz;
        s = sFrequency(iFreq);
        // To do: Band edge check
        if (targetVFO == 0) {
            updateVFOA(s);
            sendCAT("FA" + s + EOC);
            // sendOnce = "FA" + s + EOC;
        }
        else if (targetVFO == 1) {
            updateVFOB(s);
            sendCAT("FB" + s + EOC);
            // sendOnce = "FB" + s + EOC;
        }
    }//GEN-LAST:event_upBtnActionPerformed

    private void dnBtnActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_dnBtnActionPerformed
        // TODO add your handling code here:
        String s = lastVFOA;
        if (targetVFO == 1)
            s = lastVFOB;
        int iFreq = Integer.valueOf(s);
        if (tuningRateCheckBox.isSelected())
            iFreq -= fastHz;
        else
            iFreq -= slowHz;
        // To do: Band edge check
        s = sFrequency(iFreq);
        if (targetVFO == 0) {
            updateVFOA(s);
            sendCAT("FA" + s + EOC);
            // sendOnce = "FA" + s + EOC;
        }
        else if (targetVFO == 1) {
            updateVFOB(s);
            sendCAT("FB" + s + EOC);
            // sendOnce = "FB" + s + EOC;
        }
    }//GEN-LAST:event_dnBtnActionPerformed

    private void rBtn20ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_rBtn20ActionPerformed
        // TODO add your handling code here:
        sendOnce = "FA" + bS20A + EOC + "FB" + bS20B + EOC;
        logActivity("20 meter band selected");
    }//GEN-LAST:event_rBtn20ActionPerformed

    private void rBtn17ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_rBtn17ActionPerformed
        // TODO add your handling code here:
        sendOnce = "FA" + bS17A + EOC + "FB" + bS17B + EOC;
        logActivity("17 meter band selected");
    }//GEN-LAST:event_rBtn17ActionPerformed

    private void rBtn15ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_rBtn15ActionPerformed
        // TODO add your handling code here:
        sendOnce = "FA" + bS15A + EOC + "FB" + bS15B + EOC;
        logActivity("15 meter band selected");
    }//GEN-LAST:event_rBtn15ActionPerformed

    private void rBtn12ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_rBtn12ActionPerformed
        // TODO add your handling code here:
        sendOnce = "FA" + bS12A + EOC + "FB" + bS12B + EOC;
        logActivity("12 meter band selected");
    }//GEN-LAST:event_rBtn12ActionPerformed

    private void rBtn10ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_rBtn10ActionPerformed
        // TODO add your handling code here:
        sendOnce = "FA" + bS10A + EOC + "FB" + bS10B + EOC;
        logActivity("10 meter band selected");
    }//GEN-LAST:event_rBtn10ActionPerformed

    private void vfoALblMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_vfoALblMousePressed
        // TODO add your handling code here:
        sendOnce = "FR0" + EOC;
        logActivity("VFO-A selected");
    }//GEN-LAST:event_vfoALblMousePressed

    private void vfoBLblMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_vfoBLblMousePressed
        // TODO add your handling code here:
        sendOnce = "FR1" + EOC;
        logActivity("VFO-B selected");
    }//GEN-LAST:event_vfoBLblMousePressed

    private void setTimeBtnActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_setTimeBtnActionPerformed
        // TODO add your handling code here:
        // Set time here
        sendOnce = "TM" + sTime() + EOC;
        // Test log file write
        // logActivity("QMX time set");
        // Test debug file write
        // printDebugData("QMX time set");
        setTimeBtn.setText("QMX time set");
        setTimeBtn.setEnabled(false);
    }//GEN-LAST:event_setTimeBtnActionPerformed

    private void logBtnActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_logBtnActionPerformed
        // TODO add your handling code here:
        if (jLogTextField.getText().length() > 0) {
            if (contactLog) {
                logContact(jLogTextField.getText());
                jLogTextField.requestFocusInWindow();
            }
            else
                logActivity(jLogTextField.getText());
        }
        jLogTextField.setText("");
    }//GEN-LAST:event_logBtnActionPerformed

    private void sdr2qmxBtnActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_sdr2qmxBtnActionPerformed
        // TODO add your handling code here:
        String s;
        if (rigctlEnabled) {
            // Always true as button is otherwise disabled
            if (rigctlVFO > 0) {
                // Frequency has been changed in the SDR
                s = Integer.toString(rigctlVFO);
                while (s.length() < 11) {
                    s = "0" + s;
                }
                sendOnce = "FA" + s + EOC; 
            }
        }
    }//GEN-LAST:event_sdr2qmxBtnActionPerformed

    
    private void Serial_EventBasedRead(SerialPort activePort) {
        activePort.addDataListener(new SerialPortDataListener(){
            @Override
            public int getListeningEvents() {return SerialPort.LISTENING_EVENT_DATA_RECEIVED;}
            @Override
            public void serialEvent(SerialPortEvent event) {
                char c;
                int ndxEOC;    
                StringBuilder sb = new StringBuilder();
                byte [] newData = event.getReceivedData();
                for (int i=0; i<newData.length; i++) {
                    c = (char) newData[i];
                    sb.append(c);
                    //debugData.setText("");  // Clear redundantly
                }
                sCommand = sCommand.concat(sb.toString());
                //debugData.setText(sCommand);
                while (sCommand.indexOf(EOC) >= 0) {
                    ndxEOC = sCommand.indexOf(EOC) + 1;
                    processCAT(sCommand.substring(0,ndxEOC));
                    sCommand = sCommand.substring(ndxEOC);                    
                }
            }
        });
    }
    
    private void sendCAT(String s) {
        // CAT end-of-command included, no EOL
        txStream = qmxPort.getOutputStream();
        try {
            txStream.write(s.getBytes("ASCII"));
        }
        catch (IOException e) {
            JOptionPane.showMessageDialog(this, e.getMessage());
        }
    }
    
    private void processCAT(String sCommand) {
        sCommand = sCommand.toUpperCase();
        //debugData.setText("");
        //debugData.setText(sCommand);
        if (sCommand.charAt(0) == UNK)
            return;
        String cc = sCommand.substring(0,2);
        if (cc.equals("FA")) {
            // VFO-A
            if (sCommand.charAt(2) == ';') {
                // Request for frequency
            }
            else {
                // Reporting frequency
                if (sCommand.charAt(13) == ';') {
                    updateVFOA(sCommand.substring(2,13));
                }
            }
        }
        else if (cc.equals("FB")) {
            // VFO-B
            if (sCommand.charAt(2) == EOC) {
                // Request for frequency
            }
            else {
                // Reporting frequency
                if (sCommand.charAt(13) == EOC) {
                    updateVFOB(sCommand.substring(2,13));
                }
            }
        }
        else if (cc.equals("FR")) {
            // Receive mode
            if (sCommand.charAt(2) == '1') {
                rxVFO = 1;
                selectVFOB();
                if (rxVFO != lastRxVFO) {
                    logActivity("VFO-B selected");
                    logAssociatedFrequency();
                    lastRxVFO = rxVFO;
                }
            }
            else {
                rxVFO = 0;
                selectVFOA();
                if (rxVFO != lastRxVFO) {
                    logActivity("VFO-A selected");
                    logAssociatedFrequency();
                    lastRxVFO = rxVFO;
                }
            }
            targetVFO = rxVFO;      // Refers to target of tuning
        }
        else if (cc.equals("FT")) {
            // Transmit mode
            if (sCommand.charAt(2) == '1')
                txVFO = 1;
            else
                txVFO = 0;
        }
        else if (cc.equals("FW")) {
            // Filter bandwidth
        }
        else if (cc.equals("ID")) {
            // Radio ID
        }
        else if (cc.equals("IF")) {
            // Transceiver information
            if (sCommand.charAt(2) == ';') {
                // Request for transceiver information
            }
            else {
                // Reporting transceiver information
                if (sCommand.charAt(36) == ';') {
                    updateVFOA(sCommand.substring(2,13));
                    // Five spaces at substring (13,18)
                    sRIT = sCommand.substring(18,23);
                    if (sCommand.charAt(23) == '1')
                        ritStatus = true;
                    else
                        ritStatus = false;
                    // xitStatus  0 (constant) at character position 24
                    // Memory bank 0 at character position 25
                    // Memory channel 00 at substring (26, 28)
                    if (sCommand.charAt(28) == 1)
                        xcvrStatus = 1;
                    else
                        xcvrStatus = 0;
                    operMode = (int) sCommand.charAt(29) - 48;
                    rcvVFO = (int) sCommand.charAt(30) - 48;
                    // Scan status always 0 at character position 31
                    if (sCommand.charAt(32) == 0)
                        simplex = true;
                    else
                        simplex = false;
                    // Tone always 0 at character position 33
                    // Tone number always 0 at character position 34
                    // Space (constant) at character position 35
                    // EOC at character position 36
                }
            }
        }
        else if (cc.equals("MD")) {
            // Operating mode
            cOperMode = sCommand.charAt(2);
            if (cOperMode == '3') {
                modeLbl.setText(" cw");
                if (cOperMode != lastOperMode) {
                    logActivity("Receive mode CW");
                    logAssociatedFrequency();
                    lastOperMode = '3';
                }
            }
            else if (cOperMode == '6') {
                modeLbl.setText("digi");
                if (cOperMode != lastOperMode) {
                    logActivity("Receive mode digi");
                    logAssociatedFrequency();
                    lastOperMode = '6';
                }
            }
        }
        else if (cc.equals("Q0")) {
            // TCXO frequency
        }
        else if (cc.equals("Q1")) {
            // Sideband
        }
        else if (cc.equals("Q2")) {
            // Same as FA
        }
        else if (cc.equals("Q3")) {
            // VOX Enable
        }
        else if (cc.equals("Q4")) {
            // Tx rise threshold
        }
        else if (cc.equals("Q5")) {
            // Tx fall threshold
        }
        else if (cc.equals("Q6")) {
            // Cycle Min parameter
        }
        else if (cc.equals("Q7")) {
            // Sample Min parameter
        }
        else if (cc.equals("Q8")) {
            // Discard parameter
        }
        else if (cc.equals("Q9")) {
            // IQ Mode
        }
        else if (cc.equals("QA")) {
            // Japanese band limits mode
        }
        else if (cc.equals("QB")) {
            // CAT timeout enable mode
        }
        else if (cc.equals("QC")) {
            // CAT timeout vallue (seconds)
        }
        else if (cc.equals("QJ")) {
            // Tx shift threshold
        }
        else if (cc.equals("RD")) {
            // Negative RIT offset amount
        }
        else if (cc.equals("RT")) {
            // RIT status
        }
        else if (cc.equals("RU")) {
            // Positive RIT offset amount
        }
        else if (cc.equals("RX")) {
            // [command] Set radio to receive mode immediately
        }
        else if (cc.equals("SP")) {
            // Split mode
            splitVFO = (sCommand.charAt(2) == '1');
            showSplit();
            if (splitVFO != lastLoggedSplitVFO) {
                if (splitVFO)
                    logActivity("Split VFO enabled");
                else
                    logActivity("Split VFO disabled");
                logAssociatedFrequency();   // Log on change only
                lastLoggedSplitVFO = splitVFO;
            }
        }
        else if (cc.equals("TQ")) {
            // Transmit state
            if (sCommand.charAt(2) == '1') {
                denoteTx();
            }
            else {
                denoteRx();
            }
        }
        else if (cc.equals("TX")) {
            // [command] Set radio to transmit mode immediately
        }
        else if (cc.equals("VN")) {
            // Firmware version
        }
        // Tuning
        if (rxVFO == txVFO)
            splitVFO = false;
        else
            splitVFO = true;
    }

    private void disableVFOA() {
        for (int i=0; i<VCOLS; i++) {
            vfoA[i].setText("");
            vfoA[i].setEnabled(false);
            vfoALbl.setText("");
            vfoALbl.setVisible(false);
        }
    }
    
    private void enableVFOA() {
        for (int i=0; i<VCOLS; i++)
            vfoA[i].setEnabled(true);        
        vfoA[2].setText(DLM);
        vfoA[6].setText(DLM);
        vfoALbl.setVisible(true);
    }
    
    private void disableVFOB() {
        for (int i=0; i<VCOLS; i++) {
            vfoB[i].setText("");
            vfoB[i].setEnabled(false);
            vfoBLbl.setText("");
            vfoBLbl.setVisible(false);
        }
    }
    
    private void enableVFOB() {
        for (int i=0; i<VCOLS; i++)
            vfoB[i].setEnabled(true);        
        vfoB[2].setText(DLM);
        vfoB[6].setText(DLM);
        vfoBLbl.setVisible(true);
    }
    
    private void hideTuning() {
        upBtn.setVisible(false);
        dnBtn.setVisible(false);
        tuningRateCheckBox.setVisible(false);
        hideSplit();
    }
    
    private void showTuning() {
        if (TxMode)     // To do: Restructure
            return;
        upBtn.setVisible(true);
        dnBtn.setVisible(true);
        tuningRateCheckBox.setVisible(true);
        showSplit();
    }
    
    private void hideSplit() {
        splitLbl.setVisible(false);
    }
    
    private void showSplit() {
        if (splitVFO) {
            splitLbl.setText("Split On");
        }
        else {
            splitLbl.setText("Split Off");
        }
        splitLbl.setVisible(true);
    }
    
    private void selectVFOA() {  // For tuning
        vfoBLbl.setBackground(new java.awt.Color(255, 255, 255));
        vfoALbl.setBackground(tuneableVFOlabelColor);
    }
    
    private void selectVFOB() {  // For tuning
        vfoALbl.setBackground(new java.awt.Color(255, 255, 255));
        vfoBLbl.setBackground(tuneableVFOlabelColor);        
    }
    
    private void hideBandButtons() {
        rBtn20.setVisible(false);
        rBtn17.setVisible(false);
        rBtn15.setVisible(false);
        rBtn12.setVisible(false);
        rBtn10.setVisible(false);
        currentBandNdx = -1;
        previousBandNdx = -1;
    }
    
    private void showBandButtons() {
        rBtn20.setVisible(true);
        rBtn17.setVisible(true);
        rBtn15.setVisible(true);
        rBtn12.setVisible(true);
        rBtn10.setVisible(true);
        currentBandNdx = -1;
        previousBandNdx = -1;
    }
    
    private void updateVFOA(String catFrequency) {
        // catFrequency is 11 character string per QMX and TS-480 specification
        // MHz
        String s = catFrequency.substring(3,11);
        if (s.equals(lastVFOA))
            return;
        lastVFOA = s;
        vfoA[0].setText(s.substring(0,1));
        vfoA[1].setText(s.substring(1,2));
        // KHz2,6));
        vfoA[3].setText(s.substring(2,3));
        vfoA[4].setText(s.substring(3,4));
        vfoA[5].setText(s.substring(4,5));
        // Hz
        vfoA[7].setText(s.substring(5,6));
        vfoA[8].setText(s.substring(6,7));
        vfoA[9].setText(s.substring(7,8));
        
        vfoALbl.setText("A");
        bandLbl.setText(" " + sBand(s));
        setSelectedRadioGroupBand();
        updateVfoABandChangeFrequency(catFrequency);
        showTuning();
        sendFrequencyToRigctl(lastVFOA);
        logTuning();    // Conditional - See command line parameters
    }
    
    private void updateVFOB(String catFrequency) {
        // sFrequency is 11 character string per QMX and TS-480 specification
        // MHz
        String s = catFrequency.substring(3,11);
        if (s.equals(lastVFOB))
            return;
        lastVFOB = s;
        vfoB[0].setText(s.substring(0,1));
        vfoB[1].setText(s.substring(1,2));
        // KHz2,6));
        vfoB[3].setText(s.substring(2,3));
        vfoB[4].setText(s.substring(3,4));
        vfoB[5].setText(s.substring(4,5));
        // Hz
        vfoB[7].setText(s.substring(5,6));
        vfoB[8].setText(s.substring(6,7));
        vfoB[9].setText(s.substring(7,8));
        
        vfoBLbl.setText("B");
        bandLbl.setText(" " + sBand(s));
        setSelectedRadioGroupBand();
        updateVfoBBandChangeFrequency(catFrequency);
        showTuning();
        logTuning();    // Conditional - See command line parameters
    }
    
    private void logTuning() {
        // Pace verbose log of frequency changes (tuning)
        if (verboseLogTuning) {
            LocalDateTime dt = LocalDateTime.now();
            if (ChronoUnit.SECONDS.between(lastLoggedFreqDT, dt) < MIN_VFO_LOG_INTERVAL)
                  return;
            if (targetVFO == 0) {
                logActivity("VFO-A tuned to " + lastVFOA + " Hz");
                lastLoggedFreqDT = dt;
            }
            else if (targetVFO == 1) {
                logActivity("VFO-B tuned to " + lastVFOB + " Hz");
                lastLoggedFreqDT = dt;
            }
        }
    }
    
    private void logAssociatedFrequency() {
        // When verbose frequency logging is not in effect,
        // record frequency associated with other event transitions
        if (verboseLogTuning)
            return;
        LocalDateTime dt = LocalDateTime.now();
        if (splitVFO) {
            if (lastVFOA.equals(lastVFOB)) {
                logActivity("VFO's tuned to " + lastVFOA + " Hz");
            }
            else {
                logActivity("VFO-A tuned to " + lastVFOA + " Hz");
                logActivity("VFO-B tuned to " + lastVFOB + " Hz");
            }
        }
        else {
            
            if (targetVFO == 0) {
                logActivity("VFO-A tuned to " + lastVFOA + " Hz");
            }
            else if (targetVFO == 1) {
                logActivity("VFO-B tuned to " + lastVFOB + " Hz");
            }
        }
        lastLoggedFreqDT = dt;
    }
    
    private void updateVfoABandChangeFrequency(String catFreq) {
        // Make current VFO-A frequency the target frequency
        // when returning to the band from another band.
        // currentBandNdx must be updated BEFORE band frequency
        if (currentBandNdx == 0) {
            bS20A = catFreq;
        }
        else if (currentBandNdx == 1) {           
            bS17A = catFreq;
        }
        else if (currentBandNdx == 2) {
            bS15A = catFreq;
        }
        else if (currentBandNdx == 3) {
            bS12A = catFreq;
        }
        else if (currentBandNdx == 4) {
            bS10A = catFreq;
        }
        if (currentBandNdx != previousBandNdx) {
            if (currentBandNdx == 0)
                logActivity("20 meter band selected");
            else if (currentBandNdx == 1)
                logActivity("17 meter band selected");
            else if (currentBandNdx == 2)
                logActivity("15 meter band selected");
            else if (currentBandNdx == 3)
                logActivity("12 meter band selected");
            else if (currentBandNdx == 4)
                logActivity("10 meter band selected");
            if (!verboseLogTuning)
                logActivity("VFO-A " + catFreq + " Hz");
            previousBandNdx = currentBandNdx;
        }
    }

    private void updateVfoBBandChangeFrequency(String catFreq) {
        // Make current VFO-B frequency the target frequency
        // when returning to the band from another band.
        // currentBandNdx must be updated BEFORE band frequency
        if (currentBandNdx == 0) {
            bS20B = catFreq;
        }
        else if (currentBandNdx == 1) {           
            bS17B = catFreq;
        }
        else if (currentBandNdx == 2) {
            bS15B = catFreq;
        }
        else if (currentBandNdx == 3) {
            bS12B = catFreq;
        }
        else if (currentBandNdx == 4) {
            bS10B = catFreq;
        }
    }

    private void denoteTx() {
        TxMode = true;
        // Visual element
        TxIndicatorTextField.setBackground(new java.awt.Color(153,0,0));  // 'T' Red
        TxIndicatorTextField.setText("T");
        // Log
        if (!lastLoggedTxMode) {
            logActivity("Start transmit");
            logAssociatedFrequency();
        }
        lastLoggedTxMode = true;
        lastRxTxTransitionDT = LocalDateTime.now();
    }
    
    private void denoteRx() {
        TxMode = false;
        // Visual element
        TxIndicatorTextField.setBackground(new java.awt.Color(0,153,0));  // 'R' Green
        TxIndicatorTextField.setText("R");
        // Log
        if (lastLoggedTxMode) {
            long delta = ChronoUnit.SECONDS.between(lastRxTxTransitionDT, LocalDateTime.now());
            if (delta > TxRxTransitionDuration) {
                logActivity(lastRxTxTransitionDT, "End transmit");
                lastLoggedTxMode = false;
            }    
        }
    }
    
    private int band(String sF) {
        int freq = Integer.valueOf(sF.substring(0, 5));
        if (freq <= UB10 && freq >= LB20) {
            if (freq >= LB10) {
                currentBandNdx = 4;
                return 10;
            }
            else if (freq <= UB12) {
                if (freq >= LB12) {
                    currentBandNdx = 3;
                    return 12;
                }
                else if (freq >= LB15) {
                    currentBandNdx = 2;
                    return 15;
                }
                else if (freq >= LB17) {
                    currentBandNdx = 1;
                    return 17;
                }
                else if (freq >= LB20) {
                    currentBandNdx = 0;
                    return 20;
                }
            }
            else {
                currentBandNdx = -1;
                return -1;
            }
        }
        return -1;
    }

    private String sBand(String sF) {
        int freq = Integer.valueOf(sF.substring(0, 5));
        if (freq <= UB10 && freq >= LB20) {
            if (freq >= LB10) {
                currentBandNdx = 4;
                return "10";
            }
            else if (freq <= UB12) {
                if (freq >= LB12) {
                    currentBandNdx = 3;
                    return "12";
                }
                else if (freq >= LB15) {
                    currentBandNdx = 2;
                    return "15";
                }
                else if (freq >= LB17) {
                    currentBandNdx = 1;
                    return "17";
                }
                else if (freq >= LB20) {
                    currentBandNdx = 0;
                    return "20";
                }
            }
            else {
                currentBandNdx = -1;
                return "  ";
            }
        }
        return "  ";
    }
      
    private void setSelectedRadioGroupBand() {
        if (currentBandNdx == 0) {
            rBtn20.setSelected(true);
        }
        else if (currentBandNdx == 1) {
            rBtn17.setSelected(true);
        }
        else if (currentBandNdx == 2) {
            rBtn15.setSelected(true);
        }
        else if (currentBandNdx == 3) {
            rBtn12.setSelected(true);
        }
        else if (currentBandNdx == 4) {
            rBtn10.setSelected(true);
        }
    }
    
    private String sFrequency(int iFreq) {
        // iFreq is in Hz
        // Return TS-480 CAT format frequency string (11 characters)
        String s = Integer.toString(iFreq);
        while (s.length() < 11) {
            s = "0" + s;
        }
        return s;
    }
    
    private String sTime() {
        // Returns HHMMSS for computer time
        String s = LocalDateTime.now().toString();
        //01234567890123456789012
        //2024-03-03T07:30:59.448
        return s.substring(11,13) + s.substring(14, 16) + s.substring(17, 19);
    }
    
    private void logActivity(String sData) {
        if (logActive && !contactLog) {
            logDT = LocalDateTime.now();
            try {
                logFileBufferedWriter.write(logDT.toString());
                logFileBufferedWriter.write(TAB);
                logFileBufferedWriter.write(sData);
                logFileBufferedWriter.newLine();
                lastLogDT = logDT;
            }
            catch (IOException e) {
                logActive = false;
            }
        }
    }

    private void logActivity(LocalDateTime atTime, String sData) {
        if (logActive && !contactLog) {
            try {
                logFileBufferedWriter.write(TAB);  // Indent to highlight possible out-of-sequence entry
                logFileBufferedWriter.write(atTime.toString());
                logFileBufferedWriter.write(TAB);
                logFileBufferedWriter.write(sData);
                logFileBufferedWriter.newLine();
                lastLogDT = LocalDateTime.now();
            }
            catch (IOException e) {
                logActive = false;
            }
        }
    }
    
    private void logContact(String callsign) {
        if (!logActive)
            return;

        String sDT = LocalDateTime.now(ZoneOffset.UTC).toString();
        String ymd = sDT.substring(0, 10);
        String t24 = sDT.substring(11, 16);
        String logEntry = callsign + TAB;
        String sF = "";
        String sB;
        
        logEntry = logEntry + ymd + TAB + t24 + TAB;
        if (txVFO == 0) {
           sF = lastVFOA; 
        }
        else if (txVFO == 1) {
           sF = lastVFOB; 
        }
        sB = sBand(sF);
        sF = sF.substring(0, 2) + "," + sF.substring(2, 5) + "." + sF.substring(5, 6);
        logEntry = logEntry + sB + "m" + TAB;
        logEntry = logEntry + "" + sF + "kHz" + TAB;
        if (lastOperMode == '3') {
            logEntry = logEntry + "CW" + TAB + "599 599" + TAB;
        }
        else if (lastOperMode == '6') {
            logEntry = logEntry + "Digital" + TAB;
        }
        try {
            logFileBufferedWriter.write(logEntry);
            logFileBufferedWriter.newLine();
            lastLogDT = LocalDateTime.now();
        }
        catch (IOException e) {
            logActive = false;
        }
    }
    
    private void printDebugData(String sData) {
        if (debugFileEnabled) {
            debugPrint.print(LocalDateTime.now().toString() + TAB);
            debugPrint.println(sData);
        }
    }

    private void toggleRigctlEnabled() {
        if (rigctlEnabled) {
            rigctlEnabled = false;
            testButton.setBackground(new java.awt.Color(255,255,255));
            // JOptionPane.showMessageDialog(this, "Rig control disabled!");
            sdr2qmxBtn.setEnabled(false);
            sdr2qmxBtn.setVisible(false);
        }
        else {
            rigctlEnabled = true;
            testButton.setBackground(new java.awt.Color(0,153,0));
            sdr2qmxBtn.setEnabled(true);
            sdr2qmxBtn.setVisible(true);
        }
    }
    
    private void sendToRigctl(String cmd) {
        if (rigctlEnabled) {
            // JOptionPane.showMessageDialog(this, "Sending |" + cmd + "| to hamlib.");
            try (Socket rcSocket = new Socket(rigctlHost, rigctlPort)) {
                rcOutput = rcSocket.getOutputStream();
                rcWriter = new PrintWriter(rcOutput, true);
                rcWriter.println(cmd);
                rcWriter.close();
                rcSocket.close();
            }
            catch (IOException ex) {
                JOptionPane.showMessageDialog(this, "Rig control connect failed!");
                rigctlEnabled = false;
            }
        }
    }

    private void sendFrequencyToRigctl(String freq) {
        rigctlVFO = Integer.parseInt(freq);
        sendToRigctl("F " + Integer.toString(rigctlVFO + freqCorrection));
    }
    
    private String newGetRigctlFrequency() {
        // This method reads a fixed-length character array with timeout
        String s = "";
        final int cCount = 8;
        char[] msg = new char[cCount];
        final int toMS = 10;
        
            try (Socket rcSocket = new Socket(rigctlHost, rigctlPort)) {
                rcSocket.setSoTimeout(toMS);
                rcOutput = rcSocket.getOutputStream();
                rcWriter = new PrintWriter(rcOutput, true);
                rcWriter.println("f");
                // Do not explicitly close rcWriter...
                rcInput = rcSocket.getInputStream();
                rcReader = new BufferedReader(new InputStreamReader(rcInput));
                rcReader.read(msg);
                for (int i=0; i<cCount; i++)
                    s = s + msg[i];
                rcReader.close();
            }
            catch (SocketTimeoutException to) {
                // Return w/o read result
                s = "time-out";
            }
            catch (IOException ex) {
                testButton.setBackground(new java.awt.Color(255,255,255));
                JOptionPane.showMessageDialog(this, "Rig control connect failed!");
                rigctlEnabled = false;
            }
        return s;
    }

    private void processFrequencyFromRigctl(String s) {
        int iFreq = Integer.parseInt(s);
        if (iFreq == rigctlVFO)
            return;
        rigctlVFO = iFreq;      // To do: Validity check
        // Process to QMX on designated event (sdr2qmxBtn action performed)
/*
        if (debugPrintCount++ < MAX_DEBUG_PRINTS)
            printDebugData('|' + s + '|');
*/
    }
            
    public Timer CAT_update_timer = new Timer(CAT_UPDATE_INTERVAL/catMsgCount,(e) -> {
        // Under development
        String sTmp;
        sendCAT(catMsgList[(catMsgNdx++ % catMsgCount)]);
        if (sendOnce.length() > 0) {
            sendCAT(sendOnce);
            sendOnce = "";
        }
        if (rigctlEnabled && (catMsgNdx % rigctlMod == 0)) {
            // Request rig control frequency and
            // update if changed
            sTmp = newGetRigctlFrequency();
            processFrequencyFromRigctl(sTmp);
        }
    });

    
    /**
     * @param args the command line arguments
     */
    public static void main(String args[]) {
        /* Set the Nimbus look and feel */
        //<editor-fold defaultstate="collapsed" desc=" Look and feel setting code (optional) ">
        /* If Nimbus (introduced in Java SE 6) is not available, stay with the default look and feel.
         * For details see http://download.oracle.com/javase/tutorial/uiswing/lookandfeel/plaf.html 
         */
        for (int i=0; i<args.length; i++) {
           if (args[i].equals("-L"))  // Case-sensitive
               enableLog = true;
           else if (args[i].equals("-V"))
               verboseLogTuning = true;
           else if (args[i].equals("-W"))
               contactLog = true;
        }
        
        try {
            for (javax.swing.UIManager.LookAndFeelInfo info : javax.swing.UIManager.getInstalledLookAndFeels()) {
                if ("Nimbus".equals(info.getName())) {
                    javax.swing.UIManager.setLookAndFeel(info.getClassName());
                    break;
                }
            }
        } catch (ClassNotFoundException ex) {
            java.util.logging.Logger.getLogger(qmxUI.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        } catch (InstantiationException ex) {
            java.util.logging.Logger.getLogger(qmxUI.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        } catch (IllegalAccessException ex) {
            java.util.logging.Logger.getLogger(qmxUI.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        } catch (javax.swing.UnsupportedLookAndFeelException ex) {
            java.util.logging.Logger.getLogger(qmxUI.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        }
        //</editor-fold>
 
        /* Create and display the form */
        java.awt.EventQueue.invokeLater(new Runnable() {
            public void run() {
                new qmxUI().setVisible(true);
            }
        });
        
        
    }

// Debug

    String randomTestFrequency() {
        // rig control SDR++ / TS2000 format
        Random x = new Random();
        return Integer.toString(14060000 + x.nextInt(10000) - 1);
    }




    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.ButtonGroup BandBtnGroup;
    private javax.swing.JTextField TxIndicatorTextField;
    private javax.swing.JLabel bandLbl;
    private javax.swing.JLabel comPortLbl;
    private javax.swing.JComboBox<String> comSelectComboBox;
    private javax.swing.JButton connectBtn;
    private javax.swing.JTextArea debugData;
    private javax.swing.JButton disconnectBtn;
    private javax.swing.JButton dnBtn;
    private javax.swing.JTextField jLogTextField;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JButton logBtn;
    private javax.swing.JPanel mainInterfacePanel;
    private javax.swing.JLabel modeLbl;
    private javax.swing.JTextField prototypeTextField;
    private javax.swing.JButton prototypeUpDnBtn;
    private javax.swing.JRadioButton rBtn10;
    private javax.swing.JRadioButton rBtn12;
    private javax.swing.JRadioButton rBtn15;
    private javax.swing.JRadioButton rBtn17;
    private javax.swing.JRadioButton rBtn20;
    private javax.swing.JButton sdr2qmxBtn;
    private javax.swing.JButton setTimeBtn;
    private javax.swing.JLabel splitLbl;
    private javax.swing.JButton testButton;
    private javax.swing.JCheckBox tuningRateCheckBox;
    private javax.swing.JButton upBtn;
    private javax.swing.JLabel vfoALbl;
    private javax.swing.JLabel vfoBLbl;
    // End of variables declaration//GEN-END:variables
}
