//  Teensy adaptation in progress - Version # will change when complete. //

/* 
 * My first Arduino program : LM  Rev.1 - 5-letter groups     
 *                                Rev.2 - Speed control       
 *                                Rev.3 - Mother of All Keyers - MOAK
 *                                1.0.3.1   Pseudo-text generator 
 *                                          keyModes (Electronic Keyer)
 *                                                   (Straight key)
 *                                          LCD Display
 *                                          Run-time option selection
 *                                          Add pseudo-call sign mode
 *                                          Sending practice mode
 *                                1.0.3.2   Add speed adjustment mode ('Select' button)            
 *                                          Splash screen gets version ID from VERSION
 *                                Rev.4   - Omitted from this version
 *                                Rev.5   - Hybrid version - October 2019
 *                                1.0.5.1   Internally generated tone, no bug emulation or connected computer features
 *                                1.0.5.2   Compiled for Teensy - Test i2c LCD
 *                                2.x       NetMOAK versions 
 *                                3.0       Touch screen, English language words, radio terms ...
 *                                          
 * Lloyd Milligan (WA4EFS) December 2017                                         
 */

#include <Wire.h>
#include <LiquidCrystal_I2C.h>
#include <ILI9341_t3.h>
#include <UTouch.h>
#include <EEPROM.h>


const boolean DEBUG = false;
const boolean lcdPresent = false;
const boolean tftPresent = true;

boolean TEST = false;       // Canned sentence (all letters and period)
boolean incNum = false;     // Include numbers in 5-character groups
                            // or in generated pseudo-text

// This section TFT and Touch
#define TFT_DC       9  // N/C
#define TFT_CS      10  // Display CS
#define TFT_RST    255  // 255 = unused, connect to 3.3V
#define TFT_MOSI    11
#define TFT_SCLK    13
#define TFT_MISO    12  // Not used (not connected)

#define T_CS 38         // Touch CS
#define T_DO 29         // Touch data out (UTouch)
#define T_DIN 30        // Touch data in (UTouch)
#define T_CLK 36        // Touch clock (UTouch)
#define T_IRQ 22        // Touch IRQ

// Calibration kludge
#define X_PIXELS 320
#define Y_PIXELS 240
#define X_MIN 0         // Empirically determined minimum X returned by getX() for display part of screen
#define Y_MIN 26        // Same for Y
#define X_MAX 230       // Etc.     
#define Y_MAX 390 
#define X_RANGE (X_MAX - X_MIN)
#define Y_RANGE (Y_MAX - Y_MIN)

ILI9341_t3 tft = ILI9341_t3(TFT_CS, TFT_DC, TFT_RST, TFT_MOSI, TFT_SCLK, TFT_MISO);
UTouch ts(T_CLK, T_CS, T_DIN, T_DO, T_IRQ);

// Standard colors
#define  BLACK           0x0000
#define BLUE            0x001F
#define RED             0xF800
#define GREEN           0x07E0
#define CYAN            0x07FF
#define MAGENTA         0xF81F
#define YELLOW          0xFFE0  
#define WHITE           0xFFFF

// Color aliases
#define TFT_BG  YELLOW
#define TFT_TXT BLUE        // Text color 1
#define TFT_ALT RED         // Text color 2
#define SLIDER_TEXT BLACK

// Additional colors (RGB)
const uint8_t LT_GREY[] =  {224, 224, 224};

// (TFT) Application constants and variables
#include "Data/radioterms.c"
#include "Data/weights.c"
#include "Data/words.c"
const int CSPERROW = 3;     // Number of callsigns per TFT row (reduce or prevent callsign wrap)
const long TDELAY = 1000;   // Touch release wait
boolean touch = false;      // Track change when touched and released
int tftMenuLevel = 0;       // Like Enum
int tftMaxLevel  = 2;
int tftLastMenuLevel = -1;
boolean subLevel = false;

const int speedRange = 30;  // 
const int minWordSpacing = 0;
const int maxWordSpacing = 100;
const int freqRange = 1023;
int selWordSpacing = 50;    // Test slider - Range 0 - 100 [Was previously volume control]
int numSpaces = 1;          // Number of Morse spaces to insert between words - Adjustable
const String WORD_SPACING_NAMES[9] = {"", "normal", "double", "", "long", "", "longer", "", "longest"};

// (TFT) GUI
uint16_t lastX = -1;
uint16_t lastY = -1;

// Buttons
const uint16_t button_width = 200;
const uint16_t button_height = 20;
const uint16_t button_x0 = 20;
const uint16_t button_y0 = 10;
const uint16_t button_radius = 5;
const uint16_t more_button_width = 100;
const uint16_t return_button_x = 220;
const uint16_t return_button_width = 80;
const uint16_t bottom_buttons_y = 210;
const uint16_t button_vert_separation = 2;
const uint16_t right_buttons_width = 80;
const uint16_t button_hor_separation = 5;

// Sliders (menu level 2)
const uint16_t sliders_xPos = 40;
const uint16_t sliders_y0 = 20;
const uint16_t sliders_yDelta = 70;
const uint16_t sliders_width = 240;
const uint16_t sliders_height = 10;
const uint16_t sliders_pointer_width = 10;
const uint16_t sliders_pointer_height = 30;
const uint16_t sliders_text_delta = 10;
const int LED_TEST = 35;

// MOAK-generated text
#define moakTftOutputColor BLUE
#define moakTftInputColor ORANGE
const uint16_t moakTftTextSize = 2;
const uint16_t moakTftTextYdelta = 20;
const uint16_t moakTftTextXinset = 20;
const int moakTftROWS =  8;   // These two constants refer to the display window for generated text
const int moakTftCOLS = 24;
uint16_t moakTftTextColor;
int moakTftCharCount = 0;
char moakTftText[moakTftROWS][moakTftCOLS];

// Miscellaneous
const String ZERO = "0";
int v3_mode = 0;            // Analogous to ptMode, pcsMode, etc. for options added in version 3

// End TFT and Touch

/*                          
const int XMITPIN = 13;     // Output - Key CPO or external device d13
*/


// Keyer                            
                            // Logical pin 13 is the only conflict with IL9341 or UTouch
                            // Change to DIO 7, pending some other resolution
const int XMITPIN =  3;     // Output - Key CPO or external device DIO 3

int timeConstant = 1020;    // Empirically determined value [Was 850 for ATmega328PU]
int sndSpeed = 9;           // Will be revalued as read from Arduino
int minSpeed = 5;           // Must be > 0 (Prevent ÷ 0 error and extend max speed)
int dotTime;                // Valued in setSpeed()
int dashTime;               // dotTime + dotTime + dotTime;
int charTime;               // dashTime;
int wordTime;               // dashTime + dashTime + dotTime;

// Audio oscillator shield has speed control trimmer potentiometer on A0
// Assign different analog pin to panel mounted speed control potentiometer
const int speedControl = 1; //  Speed control potentiometer A1
int lastSpeed = sndSpeed;   //  For detecting change in speed setting

// Tone control  - Backed into V.1.0.4.1 from version 2 - Microcontroller generated tone (shaped externally)
const int toneControl = A2; //  Tone potentiometer wafer connects here
const int minFreq  =   200; //  Minimum tone frequency - Set to lowest tone desired
const int maxFreq  =  2000; //  Maximum tone frequency - Set to hightest tone desired
const int TONE_OUT =     8; //  DIO pin 8 - Generated tone is squarewave
int       selFreq  =   440; //  Selected tone frequency

// Test
String testString = "The quick brown fox jumped over the lazy dog's back.";

// Additional MOAK constants and variables (rev. 3)
const String VERSION = "3.0";
const int MAXWORDLENGTH = 15;  // wlen[] dimension 
const int GENWORDLENGTH = 9;   // This value must not exceed MAXWORDLENGTH.
const int NUMALPHA   = 26;     // Number of alphabetic characters (letters)
const int NUMVOWELS  = 5;      // Number of vowels
const int NUMCONSTS  = NUMALPHA - NUMVOWELS;
const int SDELAY = 500;        // Delay between sentences in pseudo-text mode

boolean ptMode = true;         // Pseudo-text mode
boolean keyMode = true;        // Overrides other modes if true
boolean pcsMode = false;       // Pseudo-callsign mode
boolean practiceMode = false;  // Practice sending by keying-in displayed text
boolean incSound = false;      // Sound out displayed text in practice mode

// Accept and process input from key or paddle
boolean straight = false;      // Straight key mode

// Arrays supporting pseudo-text generation
/*
//  Word length weights based on lexical data -
long wLen [MAXWORDLENGTH] = {52L, 1751L, 13542L, 38567L, 80482L, 138788L, 205095L,
    268755L, 322641L, 363623L, 391129L, 408435L, 418761L, 424608L, 427895L};
*/
// Made-up (arbitrary) word length weights
long wLen [MAXWORDLENGTH] = {5, 15, 35, 55, 75, 90, 105, 115, 125, 130, 135, 139, 142, 144, 145};

// Based on lexical data - first letter, letter, vowel, and consonant weights
long flWeight[NUMALPHA] = {60593L, 126028L, 213114L, 265556L, 301422L, 339834L,
    379183L, 421634L, 451726L, 463127L, 487414L, 525509L, 593554L, 618349L, 640923L,
    712575L, 717273L, 765491L, 869532L, 920780L, 942190L, 959583L, 988103L, 989754L,
    994943L, 1000000L}; 
long lWeight[NUMALPHA] = {2865L, 3394L, 4586L, 5946L, 10398L, 11254L, 11920L,
    13721L, 16418L, 16475L, 16668L, 18118L, 19013L, 21591L, 24314L, 25075L, 25118L,
    27356L, 29677L, 32982L, 33955L, 34330L, 34927L, 35011L, 35604L, 35636L};
long vWeight[NUMVOWELS] = {2865L, 7317L, 10014L, 12737L, 13710L};
long cWeight[NUMCONSTS] = {529L, 1721L, 3081L, 3937L, 4603L, 6404L, 6461L,
    6654L, 8104L, 8999L, 11577L, 12338L, 12381L, 14619L, 16940L, 20245L,
    20620L, 21217L, 21301L, 21894L, 21926L};
    
char alpha[NUMALPHA] = {'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J',
    'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
    'Y', 'Z'};
char vowel[NUMVOWELS] = {'A', 'E', 'I', 'O', 'U'};
char consonant[NUMCONSTS] = {'B', 'C', 'D', 'F','G', 'H', 'J', 'K', 'L',
    'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'V', 'W', 'X', 'Y', 'Z'};
    
const int CPCT = 80;    // Percent of pseudo-words followed by comma
const int DPCT = 20;    // Percent of pseudo-words followed by dash
const int QPCT = 10;    // Percent of pseudo-questions [sentences ending in '?']
const int NPCT = 10;    // Percent of pseudo-words that are numeric
const int MAXNLEN = 5;  // Maximum length of numeric pseudo-word (minimum is 1)
// Maximum pseudo-sentence length (number of words - minimum sentence length)
const int MAXSLEN = 12;
const int MINSLEN = 5;
// Same for punctuation delimited pseudo-phrases
const int MAXPLEN = 4;
const int MINPLEN = 2;
// Miscellaneous punctuation (convenience aliases)
const char PERIOD = '.';
const char QUESTIONMARK = '?';
const char COMMA  = ',';
const char DASH   = '-';
const char SPACE  = ' ';

// Keyer related -                  
const int DOTPIN  = 6;  // Dot side of key paddle d6
const int DASHPIN = 7;  // Dash side of key =or= straight key d7

// [Serial] LCD support [Analog pins 4 (SDA) and 5 (SCL)]
const int ROWS = 4;               // If 16x2 change ROWS to 2
const int COLS = 20;              // If 16x2 change COLS to 16
LiquidCrystal_I2C lcd(0x27, COLS, ROWS);    // Instantiate

// Other LCD-related
const int DIM = ROWS * COLS;      // One-dimensional equivalent (convenience)
const int SPLASHDURATION = 5000;  // milliseconds
boolean displayCode = false;      // Display generated Morse code on LCD
long currentCell = 0;             // Top left to bottom right display cell

// General -

// Splash screen gets version ID from the VERSION constant (top of sketch) 
const String splash21 = "  MOAK " + VERSION + "  ";
const String splash22 = "     Welcome    ";

const String splash41 = "                    ";   // Welcome screen for 4 row display
const String splash42 = "Mother Of All Keyers";
const String splash43 = "  Version " + VERSION + "   ";
const String splash44 = "                    ";

// One-dimensional copy of LCD display, upper left to lower right
char lcdCells[DIM] = "";                          // Scratch array for ticker

// Option selection
const int OPTBTN  = 4;  // Option button d4
const int SELBTN  = 5;  // Select button d5

const int NUMOPT = 9;   // Number of options

const String optList[NUMOPT] = {"Electronic Key",
                                "Straight Key",
                                "5 Alpha Groups",
                                "5 Alpha-num Grps",
                                "Rand Pseudo-text",
                                "AlpNumPseudo-txt",
                                "Pseudo callsigns",
                                "Sending Practice",
                                "Listen and send"
                               };
int currentOption = 0;        // Select button steps through options
boolean commandMode = false;  // true if and only if Option button pressed
                              // and select button not yet pressed.

// Sending practice options                 
String keyinThis = "";        // For display and comparison

// Miscellaneous
const unsigned long ONESEC = 1000L;
const unsigned long TWOSEC = 2000L;

void setup() {

  pinMode(XMITPIN, OUTPUT);
  pinMode(LED_BUILTIN, OUTPUT);     // Visual indicator
  pinMode(TONE_OUT, OUTPUT);        // Version 1.0.4.1 (and V2) - AF square wave out
  pinMode(DOTPIN, INPUT);
  pinMode(DASHPIN, INPUT);
  digitalWrite(XMITPIN, LOW);       // key up
  digitalWrite(LED_BUILTIN, LOW);   // LED off

  if (lcdPresent)
    lcd.init();

  if (tftPresent) {
    pinMode(LED_TEST, OUTPUT);
    digitalWrite(LED_TEST, HIGH);
    tft.begin();
    tft.setRotation (1);
    ts.InitTouch(PORTRAIT);
    ts.setPrecision(PREC_MEDIUM);   
  }
  
  if (DEBUG) {
    Serial.begin(9600);
    Serial.print("Mother of all keyers - Version ");
    Serial.println(VERSION);
    Serial.println();
    Serial.println("Test mode:");
    Serial.println();
    debug();
  }

  if (lcdPresent) {
    lcd.backlight();
    displaySplash();
    delay(SPLASHDURATION);
    tickerClear();
  }
  delay(ONESEC); // Dark before displaying current option
  // Afterthoughts ...
  if (lcdPresent) {
    displayCurrentOption();
    delay(TWOSEC);
    myClear();
    if (displayCode) {
      lcd.backlight();  // Turn back on for code display
    }
    else lcd.noBacklight();
  }
  else delay(ONESEC);
  
  // Random number generator gets re-seeded later, based on button press
  randomSeed(analogRead(7));  // Initialize random number generator
                              // (floating analog pin)

  if (keyMode) {
    setSpeed();               // For startup in keyer mode!
                              // Otherwise keyer defaults to initial sndSpeed
  }
  setFreq();                  // Sense tone control (overrides default selFreq)
                              // TFT will override (below)
  if (tftPresent) {
    flashLED();               // Test the LED indicator here
    tftHome();
    delay(TWOSEC);
    varsReadEEPROM();
    tftMenuLevel = 1;
    tftUpdateScreen();
//  Next does not work. T_IRQ may be out of the function's range.
//  Teensy is not listed in the documentation and 21 is the highest
//  pin# to which an interrupt can be attached for any board (Mega2560).
//  Or possibly a library ISR is already attached to T_IRQ.
//  attachInterrupt(T_IRQ, touchISR, CHANGE);
    v3_mode = 0;
    keyMode = true;
  }

} // End setup()

void loop() {
  if (tftPresent)
    tftProcessTouch();
  else {
    senseButtonPress();                    // Physical button press
    setFreq();                             // Sense tone potentiometer
    if (! lcdPresent)                      // Acquire speed from potentiometer without button press (no TFT, no LCD)
      setSpeed();
  }
  // Key input has highest priority
  if (keyMode) {
    handleKey();
    return; // Key mode
  }
  // Fall through here if NOT keymode
  if (ptMode) {                            // Pseudo-text modes
    handlePtMode();
    return;
  }
  if (pcsMode) {                           // Pseudo callsigns
    handlePcsMode();
    return;
  }
  if (practiceMode) {                      // Sending practice modes
    practiceSending();
    return;
  }
  if (v3_mode == 1) {
    outputString(rndews());
    return;
  }
  if (v3_mode == 2) {
    outputString(rndrts());
    return;
  }
  if ((currentOption == 2 || currentOption == 3))
    outputString(fiveCharacterGroup() + SPACE);
}

void handleKey() {                          // Paddle or straight key (but not bug)
  if (straight) {                           // Straight key
    if (digitalRead(DASHPIN) == LOW) {      // Key DOWN
      tone(TONE_OUT,selFreq);               // LED on
      digitalWrite(XMITPIN, HIGH);          // transmit on (if different d-pin than LED)
    }
    else {                                  // Key up
      noTone(TONE_OUT);                     // LED off
      digitalWrite(XMITPIN, LOW);           // transmit off       
    }
    return;
  }

  if (digitalRead(DOTPIN) == LOW) {         // Paddle DOT side
    dot();
  }
  else if (digitalRead(DASHPIN) == LOW) {   // Paddle DASH side
    dash();
  }  
}

void handlePtMode() {
  if (incNum) {
    outputString(rndpsn());
    if (tftAbort()) return;
  }  
  else {
    outputString(rndps());
    if (tftAbort()) return;
  }
  myDelay(SDELAY);
  tickerAddCharacter(SPACE);                // Else no space between sentences
  return;
}

void handlePcsMode() {
  // For TFT format CSPERROW per display row
  if (tftPresent) {
    String s = "";
    for (int i=0; i<CSPERROW; i++) {
      s.concat(rndpcs());
      s.concat(SPACE);
      if (tftAbort()) return;
    }
    outputString(tftFormat(s));
  }
  else
  outputString(rndpcs() + SPACE);
  if (tftAbort()) return;
  myDelay(SDELAY);
  return;
}

String fiveCharacterGroup() {
  byte b;
  char c;
  String s = "";
  
  for (int i=0; i<5; i++) {
    if (commandMode) break;
    if (incNum) {
      b = random(36); // Can be changed to < 36 for partial numbers
      if (b < 26) b = 65 + b;
      else b = 22 + b;
    }
    else b = 65 + random(26);
    c = (char) b;
    s.concat(c);
  }
  return s;
}

void dot() {
  if (tftAbort()) return;
  digitalWrite(LED_BUILTIN, HIGH);   // LED on
  digitalWrite(XMITPIN, HIGH);       // key down (if different d-pin from LED)
  tone(TONE_OUT,selFreq);            // V.1.0.4.1 (backed in from V2)
  myDelay(dotTime);
//if (tftAbort()) return;            // Do not abort during key down, else tone stays on!
  digitalWrite(LED_BUILTIN, LOW);    // LED off
  digitalWrite(XMITPIN, LOW);        // key up
  noTone(TONE_OUT);
  myDelay(dotTime);
}

void dash() {
  if (tftAbort()) return;
  digitalWrite(LED_BUILTIN, HIGH);   // LED on
  digitalWrite(XMITPIN, HIGH);       // key down
  tone(TONE_OUT,selFreq);            // V.2
  myDelay(dashTime);
//if (tftAbort()) return;            // Do not abort during key down, else tone stays on!
   digitalWrite(LED_BUILTIN, LOW);   // LED off
  digitalWrite(XMITPIN, LOW);        // key up
  noTone(TONE_OUT);
  myDelay(dotTime);
}

void outputString(String str) {
  boolean repeatSpace = false;
  char chr;
  int i;
  str.toLowerCase();
  for (i = 0; i < str.length(); i ++) {
    if (tftAbort()) return;
    if (!tftPresent)
      if (commandMode) break;
    chr = str.charAt(i);
    if (chr == SPACE) {
      if (!repeatSpace)
        for (int j=0; j<numSpaces; j++)   // Not wrapping in (tftPresent)
          delay(wordTime);
      repeatSpace = true;
      if (tftPresent) {
        tftTickerAddCharacter(SPACE);
      }
      else {
        tickerAddCharacter(SPACE);
      }
    }
    else {
      if (tftAbort()) return;
      repeatSpace = false;
      outputMorseChar(chr);
      if (tftAbort()) return;
      if (tftPresent) {
        tftTickerAddCharacter(chr);
      }
      else
        tickerAddCharacter(chr);
      senseButtonPress();
    }
  }
}

void outputMorseChar(char chr) {
  switch (chr) {
    case 'a' : dot(); dash(); break;
    case 'b' : dash(); dot(); dot(); dot(); break;
    case 'c' : dash(); dot(); dash(); dot(); break;
    case 'd' : dash(); dot(); dot(); break;
    case 'e' : dot(); break;
    case 'f' : dot(); dot(); dash(); dot(); break;
    case 'g' : dash(); dash(); dot(); break;
    case 'h' : dot(); dot(); dot(); dot(); break;
    case 'i' : dot(); dot(); break;
    case 'j' : dot(); dash(); dash(); dash(); break;
    case 'k' : dash(); dot(); dash(); break;
    case 'l' : dot(); dash(); dot(); dot();; break;
    case 'm' : dash(); dash(); break;
    case 'n' : dash(); dot(); break;
    case 'o' : dash(); dash(); dash(); break;
    case 'p' : dot(); dash(); dash(); dot(); break;
    case 'q' : dash(); dash(); dot(); dash(); break;
    case 'r' : dot(); dash(); dot(); break;
    case 's' : dot(); dot(); dot(); break;
    case 't' : dash(); break;
    case 'u' : dot(); dot(); dash(); break;
    case 'v' : dot(); dot(); dot(); dash(); break;
    case 'w' : dot(); dash(); dash(); break;
    case 'x' : dash(); dot(); dot(); dash(); break;
    case 'y' : dash(); dot(); dash(); dash(); break;
    case 'z' : dash(); dash(); dot(); dot(); break;
    case '0' : dash(); dash(); dash(); dash(); dash(); break;
    case '1' : dot(); dash(); dash(); dash(); dash();  break;
    case '2' : dot(); dot(); dash(); dash(); dash();  break;
    case '3' : dot(); dot(); dot(); dash(); dash();  break;
    case '4' : dot(); dot(); dot(); dot(); dash();  break;
    case '5' : dot(); dot(); dot(); dot(); dot(); break;
    case '6' : dash(); dot(); dot(); dot(); dot(); break;
    case '7' : dash(); dash(); dot(); dot(); dot(); break;
    case '8' : dash(); dash(); dash(); dot(); dot(); break;
    case '9' : dash(); dash(); dash(); dash(); dot(); break;
    case ',' : dash(); dash(); dot(); dot(); dash(); dash(); break;
    case '.' : dot(); dash(); dot(); dash(); dot(); dash(); break;
    case '?' : dot(); dot(); dash(); dash(); dot(); dot(); break;
    case '/' : dash(); dot(); dot(); dash(); dot(); break;
    default : dash(); dot(); dot(); dot(); dash();
  }
  if (tftAbort()) return;
  delay(charTime);
}

char decodeMorseChar(int hcode) {    // Inverse of outputMorseChar()
  switch (hcode) {
    case 12     : return 'a';
    case 2111   : return 'b';
    case 2121   : return 'c';
    case 211    : return 'd';
    case 1      : return 'e';
    case 1121   : return 'f';
    case 221    : return 'g';
    case 1111   : return 'h';
    case 11     : return 'i';
    case 1222   : return 'j';
    case 212    : return 'k';
    case 1211   : return 'l';
    case 22     : return 'm';
    case 21     : return 'n';
    case 222    : return 'o';
    case 1221   : return 'p';
    case 2212   : return 'q';
    case 121    : return 'r';
    case 111    : return 's';
    case 2      : return 't';
    case 112    : return 'u';
    case 1112   : return 'v';
    case 122    : return 'w';
    case 2112   : return 'x';
    case 2122   : return 'y';
    case 2211   : return 'z';
    case 22222     : return '0';
    case 12222     : return '1';
    case 11222     : return '2';
    case 11122     : return '3';
    case 11112     : return '4';
    case 11111     : return '5';
    case 21111     : return '6';
    case 22111     : return '7';
    case 22211     : return '8';
    case 22221     : return '9';
    case 21112     : return '-';
    case 21121     : return '/';
    case 41     : return ',';
    case 42     : return '.';
    case 43     : return '?';
    default     : return '*';
  }
}

int pHash(String mc) {                        // Punctuation hash
  // Supplement to decodeMorseChar()
  if (mc == "221122") return 41;
  if (mc == "121212") return 42;
  if (mc == "112211") return 43;
  return mc.toInt();  
}

void setSpeed() { // Based on input from speed potentiometer
  sndSpeed = analogRead(speedControl) / 32 + minSpeed; // 0-1023 to w.p.m.
  dotTime = timeConstant / sndSpeed;
  dashTime = dotTime + dotTime + dotTime;
  charTime = dashTime;
  wordTime = dashTime + dashTime + dotTime;
  if (sndSpeed != lastSpeed) {
    lastSpeed = sndSpeed;
    if (lcdPresent) {
      displayLCD("Setting speed to", (String) sndSpeed + " wpm ...");
      delay(TWOSEC);
      if (not displayCode)
        lcd.noBacklight();
      myClear();
    }
  }
}

/*
 * Rev.3 MOAK Utilities
 */

boolean isVowel(char c) { // true if and only if c is a vowel
  for (int i=0; i<NUMVOWELS; i++)
    if (c == vowel[i]) return true;
  return false;
}

int rndwl() {   // Return a weighted random word length
  int len;
  long r = (long) random(wLen[GENWORDLENGTH-1]);
  for (len=0; len<GENWORDLENGTH; len++)
    if (r < wLen[len]) break;
  return ++len;
}

char rndfl() {  // Random FIRST letter
  int i;
  long r = (long) random(flWeight[NUMALPHA-1]);
  for (i=0; i<NUMALPHA; i++)
    if (r < flWeight[i]) {
      return alpha[i];
    }
  return '*'; // Should be unreachable
}

char rndl() {   // Random letter
  int i;
  long r = (long) random(lWeight[NUMALPHA-1]);
  for (i=0; i<NUMALPHA; i++)
    if (r < lWeight[i]) return alpha[i];
  return '!';   // Should be unreachable 
}

char rndc() {   // Random consonant
  int i;
  long r = (long) random(cWeight[NUMCONSTS-1]);
  for (i=0; i<NUMCONSTS; i++)
    if (r < cWeight[i]) return consonant[i];
  return '@';   // Should be unreachable 
}

char rndv() {   // Random vowel
  int i;
  long r = (long) random(vWeight[NUMVOWELS-1]);
  for (i=0; i<NUMVOWELS; i++)
    if (r < vWeight[i]) return vowel[i];
  return '#';   // Should be unreachable 
}

char rndpd() {  // Random phrase delimiter
  // Revise to incorporate additional delimiters as desired
  int r = random(CPCT+DPCT);
  if (r < CPCT)
    return COMMA;
  return DASH;  
}

String rndnum() { // Random numeric string (pseudo-word) satisfying MAXNLEN
  long r = (long) random(pow(10, random(MAXNLEN)+1)) + 1;
  return String(r);  
}

String rndpw()  {     // Generate random pseudo-word
  int len = rndwl();
  if (len == 1) {
    if (random(2) == 0) return "I";
    else return "A";
  }
  char chr = rndfl();
  boolean v = isVowel(chr);
  String word = "";
  word.concat(chr);
  for (int i = 2; i <= len; i++) {
    if (i%3 == 1) {
      chr = rndl();
      v = isVowel(chr);
      word.concat(chr);
      continue;
    }
    else {
      if (v) word.concat(rndc());
      else word.concat(rndv());
      v = not(v);      
    }
  }
  return word;
}

String rndps() {  // Random pseudo-sentence
  char punc;
  char lp = PERIOD;
  int sl = random(MAXSLEN) + MINSLEN;
  int pl = random(MAXPLEN) + MINPLEN;
  String s = "";
  if (random(100) < QPCT) lp = QUESTIONMARK;
  for (int i=1; i<sl; i++) {
    s.concat(rndpw());
    // Phrase punctuation
    if (i%pl == 0) {
      punc = rndpd();
      if (punc == DASH) s.concat(SPACE);
      s.concat(punc);
      pl = random(MAXPLEN) + MINPLEN;   // Phrase lengths should vary within sentence
    }
    s.concat(SPACE);
  }
  s.concat(rndpw());
  s.concat(lp);
  s.concat(SPACE);
  if (tftPresent) {                     // TFT cosmetics
    // Modify s for line wrap, etc.
  }
  if (tftPresent)
    return tftFormat(s);
  else
    return s;
}

String rndpsn() { // Random pseudo-sentence including numbers
  char punc;
  char lp = PERIOD;
  int sl = random(MAXSLEN) + MINSLEN;
  int pl = random(MAXPLEN) + MINPLEN;
  String s = "";
  if (random(100) < QPCT) lp = QUESTIONMARK;
  String rn = rndnum();
  String pw;
  for (int i=1; i<sl; i++) {
    if (random(100) < NPCT) {
      pw = rn;
      rn = rndnum();
    }
    else pw = rndpw();
    s.concat(pw);
    // Phrase punctuation
    if (i%pl == 0) {
      punc = rndpd();
      if (punc == DASH) s.concat(SPACE);
      s.concat(punc);
      pl = random(MAXPLEN) + MINPLEN;
    }
    s.concat(SPACE);
  }
  s.concat(rndpw());
  s.concat(lp);
  s.concat(SPACE);
  if (tftPresent)
    return tftFormat(s);
  else
    return s;
}

// Callsign generating function replaced by the same-named segmented method below
/* 
String rndpcs() {                   // Random pseudo-call sign
  // Made-up rules similar to real call signs
  String cs = "";
  char c;
  if (random(10) < 9) {             // Starts with a letter
    c = alpha[random(NUMALPHA)];
    cs.concat(c);
    if ((c == 'K') or (c == 'W')) { // US K or W callsign
      if (random(10) < 5)           // Second alpha half the time
        cs.concat(alpha[random(NUMALPHA)]);
      }
    if ((c == 'A') or (c == 'K') or (c == 'N') or (c == 'W')) { // US any
      cs.concat(random(10));        // Number between 0 and 9
      cs.concat(alpha[random(NUMALPHA)]);     // At least one more
      if (random(10) < 5)
        cs.concat(alpha[random(NUMALPHA)]); 
      if (random(100) < 95)
        cs.concat(alpha[random(NUMALPHA)]);
      return cs;      
    }
    // Possibly a second letter here followed by a single number
    // or no second letter and a one- or two-digit number
    if (random(10) < 5) {
      cs.concat(alpha[random(NUMALPHA)]);
      cs.concat(random(10));
    }
    else {
      cs.concat(random(90) + 10);
    }
    // then one to three letters (for now add two letters)
      cs.concat(alpha[random(NUMALPHA)]);
      cs.concat(alpha[random(NUMALPHA)]);
      return cs;
  }
  else  {                               // Starts with a number
    cs.concat(random(9) + 1);
    cs.concat(alpha[random(NUMALPHA)]); // Must be a letter!
    // Another number here
    cs.concat(random(9) + 1);
    // then one or two letters
    cs.concat(alpha[random(NUMALPHA)]);
    if (random(10) < 5)
      cs.concat(alpha[random(NUMALPHA)]);
    return cs;
  }
}
*/

String rndpcs() {                   // Random pseudo-call sign
  String cs = "";
  if (random(100) < 5) {            // Proportion with a DX prefix before callsign
    cs.concat(rndDXpx());
    cs.concat('/');
  }
  if (random(10) < 4) {             // Proportion of US callsigns
    cs.concat(rndUSpx());
    if (random(10) < 1)             // Proportion of 1-character suffixes (US calls)
      cs.concat(rndAlphaSuffix(1));
    else if (random(10) < 5)        // Proportion of remainder that are 2-letter calls
      cs.concat(rndAlphaSuffix(2));
    else
      cs.concat(rndAlphaSuffix(3));
  }
  else {
    cs.concat(rndDXpx());
    if (random(10) < 2)             // Proportion of 1-character suffixes
      cs.concat(rndAlphaSuffix(1));
    else if (random(10) < 8)        // Proportion of remainder that are 2-letter calls
      cs.concat(rndAlphaSuffix(2));
    else
      cs.concat(rndAlphaSuffix(3));    
  }
  return cs;
}

String rndUSpx() {                    // Random US (and territory) prefix
  const char firstLetter[4] = "KWAN";
  String prefix = "";
  if (random(10) < 8) {               // Start with K or W
    prefix.concat(firstLetter[random(2)]);  
    if (random(10) < 4)               // Append second letter
      prefix.concat(alpha[random(26)]);
  }
  else {                              // Start with A or N (no second letter)
    prefix.concat(firstLetter[random(2)+2]);  
  }
  prefix.concat(random(10));          // Always append a single digit 0 - 9
  return prefix;
}

String rndDXpx() {                    // Random DX prefix
  const char firstLetter[22] = "BCDEFGHIJLMOPQRSTUVXYZ";
  String prefix = "";
  if (random(10)) {                   // Start with a letter
    prefix.concat(firstLetter[random(22)]);
    prefix.concat(random(10));        // Append a single digit
    if ((random(10) < 2) and (prefix.charAt(1) != '0'))
      prefix.concat(random(10));      // And sometimes another
  }
  else {                              // Start with a number
    prefix.concat(random(9)+1);
    prefix.concat(alpha[random(26)]); // Next must be alpha
    prefix.concat(random(10));        // Another numeric digit
  }
  return prefix;
}

String rndAlphaSuffix(int len) {      // Random alpha suffix of specified length
  String suffix = "";
  for (int i=0; i<len; i++)
    suffix.concat(alpha[random(26)]);
  return suffix;
}

void displaySplash() {
  if (not lcdPresent) return;         // Not relevant (Redundant check)
  myClear();
  if (ROWS == 2) {
    lcd.setCursor(0,0);
    lcd.print(splash21);
    lcd.setCursor(0,1);
    lcd.print(splash22);
  }
  else if (ROWS == 4) {
    lcd.setCursor(0,0);
    lcd.print(splash41);
    lcd.setCursor(0,1);
    lcd.print(splash42);
    lcd.setCursor(0,2);
    lcd.print(splash43);
    lcd.setCursor(0,3);
    lcd.print(splash44);
  }
  return;
}

void tickerClear() {
  if (not lcdPresent) return;        // Not relevant
  for (int i=0; i<ROWS; i++)
    for (int j=0; j<COLS; j++)
      lcdCells[i*COLS + j] = SPACE;  
}

void tickerAddCharacter(char c) {    // Add character to lcdCells array
  if (not displayCode) return;       // Not relevant
  if (commandMode) return;
  if (practiceMode) return;          // Do not duplicate display in practice mode
  // Shift left and append character c to last cell of last row
  for (int i=0; i<ROWS; i++)
    for (int j=0; j<COLS; j++) {
      int k = i*COLS + j;
      if (k+1 < DIM) lcdCells[k] = lcdCells[k+1];
    }
  lcdCells[DIM-1] = c;
  // forwardDisplay(c);             // Fills screen then ticks characters
  vScrollDisplay(c);                // Fills screen then scrolls lines
  return;
}

//  (LCD) Following functions have been superceded by vScrollDisplay()
//  Begin at lower-right corner; shift left and up
/*
void tickerDisplay() {              // Copy lcdCells array to LCD
  // Context pre-checked in tickerAddCharacter
  for (int i=0; i<ROWS; i++) {
    for (int j=0; j<COLS; j++) {
      lcd.setCursor(j,i);
      lcd.print(lcdCells[i*COLS + j]);
    }
  }
}
*/

//  (LCD) Paint upper left to lower right until full, then continue as ticker
/*
void forwardDisplay(char c) {
  // Context pre-checked in tickerAddCharacter
  if (currentCell < DIM) {
    int row = currentCell / COLS;
    int col = currentCell % COLS;
    lcd.setCursor(col, row);
    lcd.print(c);
    currentCell++;
  }
  else tickerDisplay();
}
*/

void vScrollDisplay(char c) {                 // Scroll display vertically
  // clear top line
  // copy remaining lines up one
  // set cursor to leftmost character of bottom line
    int row = currentCell / COLS;
    int col = currentCell % COLS;
  if (currentCell < DIM) {                    // First screen (not full)
    lcd.setCursor(col, row);
    lcd.print(c);
  }
  else {
    if (col == 0) {
      for (int i = 1; i < ROWS; i++)
        for (int j = 0; j < COLS; j++) {      // Copy rows up
          lcd.setCursor(j, i-1);
          lcd.print(lcdCells[i*COLS + j - 1]);
        }
      for (int j = 0; j < COLS; j++)   {      // Clear bottom row
        lcd.setCursor(j, ROWS-1);
        lcd.print(SPACE);
      }
    }
    lcd.setCursor(col, ROWS-1);               // Bottom row (from left)
    lcd.print(c);
  }
  currentCell++;    
}

void senseButtonPress() {           // Option and Select buttons
  randomSeed(millis());             // Better than sensing an analog pin
  if (tftPresent) {
    tftProcessTouch();              // Alternate option selection
    return;
  }
  if (not lcdPresent) return;       // LCD-based option selection
  if (digitalRead(OPTBTN) == HIGH)
    processOptionButton();
  else if (digitalRead(SELBTN) == HIGH)
    processSelectButton();
}

void processOptionButton() {
  if (tftPresent)
    return;
  commandMode = true;
  displayCurrentOption();
  delay(ONESEC);          // Minimum time in commandMode (software debounce)
  currentOption = ++currentOption % NUMOPT;
}

void processSelectButton() {
  // Consider restructuring as option tree

  if (tftPresent)             // No 'Select' button -- should not reach here
    return;

  if (commandMode) {
    currentOption = (--currentOption + NUMOPT) % NUMOPT;
    // To do: generic display utility - call with simple message
    if (lcdPresent)
      displayLCD("Setting option:",optList[currentOption]);
    delay(ONESEC);
    int i = currentOption;    // Less typing!
    // Process each option type
    if (i == 0) {             // Electronic keyer
      keyMode = true;
      straight = false;      
      displayCode = false;    // Kludge to turn-of backlight
    }
    else if (i == 1) {        // Straight key
      keyMode = true;
      straight = true;
      displayCode = false;
    }
    else if (i == 2) {        // 5-letter groups with LCD display
      keyMode = false;
      ptMode = false;
      pcsMode = false;
      incNum = false;      
      practiceMode = false;
      displayCode = true;
    }
    else if (i == 3) {        // 5-character alphanumeric groups, with LCD
      keyMode = false;
      ptMode = false;
      pcsMode = false;
      practiceMode = false;
      incNum = true;
      displayCode = true;            
    }
    else if (i == 4) {        // Pseudo-text, with LCD
      keyMode = false;
      ptMode = true;
      pcsMode = false;
      practiceMode = false;
      incNum = false;
      displayCode = true;            
    }
    else if (i == 5) {        // Pseudo-text + numbers, with LCD
      keyMode = false;
      ptMode = true;
      pcsMode = false;
      practiceMode = false;
      incNum = true;
      displayCode = true;                
    }
    else if (i == 6) {        // Pseudo call signs with LCD
      keyMode = false;
      straight = false;
      ptMode = false;
      practiceMode = false;
      pcsMode = true;
      displayCode = true;                      
    }
    else if (i == 7) {        // Practice sending by keying-in displayed text
      keyMode = false;
      straight = false;
      ptMode = false;
      pcsMode = false;
      practiceMode = true;
      incSound = false;       // Relevant only in sending practice modes
      displayCode = true;                      
    }
    else if (i == 8) {        // Practice sending by keying-in sounded and displayed text
      keyMode = false;
      straight = false;
      ptMode = false;
      pcsMode = false;
      practiceMode = true;
      incSound = true;        // Sound pattern to copy
      displayCode = true;                      
    }
    // Additional options go here
    
    // Regardless of whether option was changed or not -
    commandMode = false;
    if (lcdPresent) {
      myClear();
      // Backlight status as required by selected option type
      if (displayCode) lcd.backlight();
      else lcd.noBacklight();
    }
  }
  else {            // NOT command mode
    // Overload Select button ... (Version 1.0.3.2)
    // If sensed when not in command mode, enter (or exit) speed setting mode
    // On entry to speed setting mode, continuously adjust speed (based on
    // potentiometer adjustments until Select button is pressed again, then
    // exit this special mode.
    processSpeedAdjustment();
  }
  // Regardless of whether button pressed in command mode or not
  // Adjust speed
  setSpeed();
  return;
}

void displayCurrentOption() {       // Meaningless to display any other option
  if (not lcdPresent) return;       // Not relevant
  displayLCD("Current option:",optList[currentOption]);
}

void myClear() {
  tickerClear();
  if (lcdPresent) {
    lcd.clear();
    lcd.setCursor(0,0);
    currentCell = 0;
  }
}

//  Most displays (except ticker) are two lines, either 16 or 20 characters.
//  Generic utility to display two lines centered vertically.

void displayLCD(String line1, String line2) {
  int row = ROWS/2 - 1; // Same as if..else
  myClear();
  lcd.setCursor(0, row);
  lcd.print(line1);
  lcd.setCursor(0, row+1);
  lcd.print(line2);
  lcd.backlight();
}

void darkenLCD(int milliseconds) {
  lcd.noBacklight();
  delay(milliseconds);
  lcd.backlight();      // Calling context implies backlight ON 
}

// Keying practice option (copy displayed string to key)

void practicePattern() {      // Text to be keyed-in for sending practice
  // Generates one of the option text types
  senseButtonPress();
  if (!tftPresent)
    if (commandMode) return;
  int r = random(4000);
  if (r < 2000) keyinThis = rndpw();          // Pseudo-word
  else if (r < 3000) keyinThis = rndpcs();    // Pseudo-callsign
  else if (r < 3500) keyinThis = rndnum();    // Numeric string
  else keyinThis = fiveCharacterGroup();      // Five character group
  return;
}

void practiceSending() {  // Display practice text and compare keyed input ...
  // Similar to 'Trainer' function by Tom Lewis http://www.qsl.net/n4tl/
  if (tftAbort()) return;
  if (!tftPresent) {
    if (commandMode) return;
    setSpeed();
  }
  char c;                                     // Translated character (English character)
  String mc = "";                             // Morse character (dot=1, dash=2)
  int mcph;                                   // Integer hash of mc
  // Experimenting here -
  unsigned long minEOC = dashTime / 2;        // Minimum inter-character time
  practicePattern();
  int patternLength = keyinThis.length();     // Length of text to be keyed-in
  int characterPosition = 0;
  int lcdRow = ROWS / 2;
  long now = (unsigned long) millis();        // For comparing elapsed times
  boolean success = false;                    // Until correctly keyed
  while (not success) {
    if (!tftPresent)
      if (commandMode) return;                // Prevent erasing answer row (Kludge)
    if (tftAbort()) return;
    keyinThis.toUpperCase();                  // Cosmetic change for presentation only
    if (tftPresent)
      displayTFT(keyinThis, "");
    else
      displayLCD(keyinThis,"");
    if (incSound)
      outputString(keyinThis);                // Sound-out text to be keyed-in
    keyinThis.toLowerCase();
    if (lcdPresent) {
      lcd.setCursor(0,lcdRow);
      lcd.print("                ");          // Erase line
      lcd.setCursor(0,lcdRow);
    }
    characterPosition = 0;
    while (characterPosition < patternLength) {
      if (tftAbort()) return;
      if (lcdPresent)
        lcd.setCursor(characterPosition, lcdRow);
      if (digitalRead(DOTPIN) == LOW) {
        dot();
        mc.concat('1');
        now = millis();
      }
      else if (digitalRead(DASHPIN) == LOW) {
        dash();
        mc.concat('2');
        now = millis();
      }
      if (mc == "") {
        if (digitalRead(OPTBTN) == HIGH)
          break;
        now = millis();
        continue;
      }
      else if (millis() - now > minEOC) {
        mcph = pHash(mc);
        mc = "";
        c = decodeMorseChar(mcph);
  
        if (c == keyinThis.charAt(characterPosition)) {
          if (tftPresent)
            tft.print(c);
          else if (lcdPresent)
            lcd.print(c);
          if (characterPosition == patternLength - 1)
            success = true;
        }
        else {
          if (tftPresent)
            tft.print(c);
          else if (lcdPresent)
            lcd.print(c);   // Display sensed character (or default '*' if not understood)
          if (tftAbort()) return;
          delay(ONESEC);
          break;
        }
        characterPosition++;
        senseButtonPress();
        now = millis();
      }
      senseButtonPress();
      if (!tftPresent)
        if (commandMode) break;
    }   // while character position < length

    senseButtonPress();
    if (!tftPresent)
      if (commandMode) break;
  }   // while not success (not complete)

  senseButtonPress();
  if (!tftPresent)
    if (commandMode) return;
  delay(ONESEC);
}

// Next added in version 1.0.3.2 -

  void processSpeedAdjustment() {
    if (not lcdPresent) return;      // Meaningless without LCD
    while (digitalRead(SELBTN) == HIGH) ;       // Wait for button press to complete
     lcd.backlight();                           // Always on in setSpeedMode
     displayLCD("Speed adjustment"," mode ...");
    // This is a tight loop - Do not sense button externally
    // commandMode disabled until Select button pressed to exit mode
    while (true) {
      if (digitalRead(SELBTN) == HIGH) {        // End setSpeedMode
        while (digitalRead(SELBTN) == HIGH) ;   // Wait for button press to complete
        break;                                  // Exit while (true) loop
      } // End if (digitalRead(SELBTN) == HIGH)
        delay (100);                            // Adjust for smooth display
        specialSetSpeed();
    } // end while (true) - tight loop
    lcd.clear();
    if (displayCode) lcd.backlight();           // Conditionally on or off when exiting set speed mode
    else lcd.noBacklight();                     // Might be redundant (check setSpeed())
  }

void specialSetSpeed() {      // Goes with processSpeedAdjustment()
  // No delay, no conditional setting of LCD display
  sndSpeed = analogRead(speedControl) / 32 + minSpeed; // 0-1023 to w.p.m.
  dotTime = timeConstant / sndSpeed;
  dashTime = dotTime + dotTime + dotTime;
  charTime = dashTime;
  wordTime = dashTime + dashTime + dotTime;
  if (sndSpeed != lastSpeed) {
    lastSpeed = sndSpeed;
    lcd.clear();
    displayLCD("Setting speed to", (String) sndSpeed + " wpm ...");
  }
}

//  Add in 1.0.4.1 (From Version 2)
void setFreq() { // From pitch potentiometer (tone control)
  selFreq = analogRead(toneControl) + 200; // 200-1223 Hz
  return;
}

//  Development in progress below this line

void debug() {
  randomSeed(analogRead(7));  // Initialize random number generator
  if (lcdPresent) {
    lcd.backlight();
    displayLCD("Test mode...","");
  }
  while (true) {
    ;                         // Test function here
  }
  return;
}

// TFT and Touch functions below this line.  Some functions are for testing only.
// Imported from Teensy_TFT-Test-U

void tftHome() {
  tftSplash();
  tftMenuLevel = 0;
}

void tftSplash() {
  tft.fillScreen(TFT_BG);
  tft.setTextColor(TFT_ALT);  
  tft.setTextSize(2);
  tft.setCursor(40, 80);
  tft.println("Mother of all Keyers");
  tft.setCursor(35, 100);
  tft.println("     version " + String(VERSION));
  tft.setTextColor(TFT_TXT);  
  tft.setCursor(30, 140);
  tft.println("https://www.lloydm.net");
}

void tftUpdateScreen() {
  if (tftMenuLevel < 0)
    tftMenuLevel = 0;
  else if (tftMenuLevel > tftMaxLevel)
    tftMenuLevel = tftMaxLevel;
  if (tftMenuLevel == tftLastMenuLevel)
    return;
  if (tftMenuLevel == 0)
    tftHome();
  else if (tftMenuLevel == 1)    // Main menu level
    tftDisplayMenu();
  else if ((tftMenuLevel == 2) && (subLevel)) {
    subLevel = false;
    tftSlidersMenu();
  }

  if (tftMenuLevel != 0)         // 
     tftDisplayReturnButton();
  tftLastMenuLevel = tftMenuLevel;
  delay (TDELAY);  
}

boolean tftAbort() {
  // Minimize time here
  if (ts.dataAvailable())
    return true;
  return false;
}

void tftProcessTouch() {
  if (!ts.dataAvailable()) {
    touch = false;
    return;
  }
  tftMainTouchProcessing();
  tftUpdateScreen();
  return;
}

void tftMainTouchProcessing() {
  // This function is called by tftProcessTouch
  ts.read();
  int x = myGetX();
  int y = myGetY();
  if (touch || (x < 0) || (y < 0))
    return;
  touch = true;
  lastX = x;
  lastY = y;
  // Screens
  int iOpt = tftOptionSelected();
  if (tftMenuLevel == 0) {
    tftMenuLevel++;
  }
  else if (tftReturnButtonPressed()) {
    // Return button pressed
    v3_mode = 0;
    tftMenuLevel--;
    if (tftMenuLevel < 0)
      tftMenuLevel = 0;
    keyMode = true;
  }
  // Buttons
  else if (tftMenuLevel == 1) {
    if (tftEnglishWordsButtonPressed()) {
      v3_mode = 1;
      tftMenuLevel = 2;
      tftProcessSelectedOption(0);
    }
    else if (tftRadioTermsButtonPressed()) {
      v3_mode = 2;
      tftMenuLevel = 2;
      tftProcessSelectedOption(0);
    }
    else if (iOpt >= 0) {
      tftMenuLevel = 2;
      tftProcessSelectedOption(iOpt);
    }
    else if (tftMoreButtonPressed()) {
      tftMenuLevel = 2;
      subLevel = true;
    }
  }
  // Sliders 
  else if (tftMenuLevel == 2) {
    tftUpdateSlider(x, y);
  }

  if (tftMenuLevel < 2) {
    tftSetDefaultMode();  // No option selected (default = key mode)
  }
  tftUpdateScreen();
}

void tftNumSpaces() {
    // 'Spaces between words' was previously a volume control, hence the artificial 0 - 100 range
    // Convert to number of spaces between 'words'
    if (selWordSpacing <= 60)
      numSpaces = 1;
    else if (selWordSpacing <= 70)
      numSpaces = 2;
    else if (selWordSpacing <= 80)
      numSpaces = 4;
    else if (selWordSpacing <= 90)
      numSpaces = 6;
    else if (selWordSpacing <= maxWordSpacing)
      numSpaces = 8;  
}

void tftUpdateSlider(uint16_t x, uint16_t y) {
  if (tftTouchedInBox(sliders_xPos, sliders_y0, sliders_width, sliders_pointer_height)) {
    // Speed control
    sndSpeed = (x - sliders_xPos) * speedRange / sliders_width + minSpeed;
    if (sndSpeed < minSpeed)
      sndSpeed = minSpeed;
    else if (sndSpeed > (minSpeed + speedRange))
      sndSpeed = minSpeed + speedRange;
    tftSetSpeed();    // dot and dash timing
    displaySpeedSlider();
    varsWriteEEPROM();
  }
  else if (tftTouchedInBox(sliders_xPos, sliders_y0 + sliders_yDelta, sliders_width, sliders_pointer_height)) {
    // Tone control
    selFreq = (x - sliders_xPos) * freqRange / sliders_width + minFreq;
    if (selFreq < minFreq)
      selFreq = minFreq;
    if (selFreq > (minFreq + freqRange))
      selFreq = minFreq + freqRange;
    displayToneSlider();
    varsWriteEEPROM();
  }
  else if (tftTouchedInBox(sliders_xPos, sliders_y0 + sliders_yDelta + sliders_yDelta, sliders_width, sliders_pointer_height)) {
    // V3 word spacing control
    selWordSpacing = (x - sliders_xPos) * maxWordSpacing / sliders_width;
    if (selWordSpacing < minWordSpacing)
      selWordSpacing = minWordSpacing;
    else if (selWordSpacing > maxWordSpacing)
      selWordSpacing = maxWordSpacing;
    tftNumSpaces();
    displayWordSpacingSlider();
    varsWriteEEPROM();
  }
}

boolean tftMoreButtonPressed() {
  return tftTouchedInBox(button_x0, bottom_buttons_y, more_button_width, button_height);
}

boolean tftReturnButtonPressed() {
  return tftTouchedInBox(return_button_x, bottom_buttons_y, return_button_width, button_height);
}

boolean tftTouchedInBox(uint16_t x, uint16_t y, uint16_t w, uint16_t h) {
  if (lastX < x) return false;
  if (lastY < y) return false;
  if (lastX > (x + w)) return false;
  if (lastY > (y + h)) return false;
  return true;
}

void displayButton(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg, String lbl, uint16_t sz) {
  tft.fillRoundRect(x, y, w, h, button_radius, bg);
  tft.setTextColor(fg);
  tft.setTextSize(sz);
  tft.setCursor(x+5, y+2);
  tft.print(lbl);
  tft.drawRoundRect(x, y, w, h, button_radius, BLACK);
  return;
}

void displaySlider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg, String lbl, uint16_t sz, int iPos, int xLbl) {
  int x0 = iPos, y0 = y - sliders_pointer_height/2 + h/2;
  tft.fillRect(x - sliders_pointer_width, y0, w + sliders_pointer_width + sliders_pointer_width, sliders_pointer_height, TFT_BG);
  tft.fillRoundRect(x, y, w, h, button_radius, bg);
  tft.fillRoundRect(x0, y0, sliders_pointer_width, sliders_pointer_height, 2, fg);
  tft.drawRoundRect(x, y, w, h, button_radius, BLACK);
  tft.drawRoundRect(x0, y0, sliders_pointer_width, sliders_pointer_height, 2, BLACK);
  tft.setTextColor(fg);
  tft.setTextSize(sz);
  tft.setCursor(xLbl, y+sliders_pointer_height-h/2);    // To do: Centering function
  tft.print(lbl);
  return;
}

void displaySlider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg, String lbl, uint16_t sz, int iPos) {
  displaySlider(x, y, w, h, fg, bg, lbl, sz, iPos, x+w/3 );
}

void displaySlider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg, String lbl, uint16_t sz) {
  displaySlider(x, y, w, h, fg, bg, lbl, sz, x + w/2);
}

void displaySlider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg, String lbl) {
  displaySlider(x, y, w, h, fg, bg, lbl, 2);
}

void displaySlider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg) {
  displaySlider(x, y, w, h, fg, bg, "");
}

// Placeholders - Will need to read and update current values
void displaySpeedSlider() {
  int iPos = (sndSpeed - minSpeed) * 8 + sliders_xPos;   // Speed range = 5 - 35, slider range = 40 - 280 -> 240 / 30 = 8
  displaySlider(sliders_xPos, sliders_y0, sliders_width, sliders_height, TFT_TXT, MAGENTA, String(minSpeed) + "      Speed      " + String(minSpeed + speedRange), 2, iPos, 40);
  displayTextSpeed();
}

void displayToneSlider() {
  long iPos = (selFreq - minFreq) * sliders_width / freqRange + sliders_xPos;
  displaySlider(sliders_xPos, sliders_y0 + sliders_yDelta, sliders_width, sliders_height, TFT_TXT, MAGENTA, "Low    Pitch    High", 2, (int) iPos, 40);
  displayTextTone();                  
}

void displayWordSpacingSlider() {
  long iPos = selWordSpacing * sliders_width / maxWordSpacing + sliders_xPos;
  displaySlider(sliders_xPos, sliders_y0 + sliders_yDelta + sliders_yDelta, sliders_width, sliders_height, TFT_TXT, MAGENTA, "Spaces between words", 2, iPos, 40);
  displayWordSpacingText();    // This works, but isn't meaningful.
}

void displayTextSpeed() {
  int xPos = (sndSpeed - minSpeed) * 8 + sliders_xPos  + sliders_pointer_width;
  const int text_width = 60;
  if (sndSpeed > (minSpeed + speedRange)/2)
    xPos = xPos - text_width; 
  tft.setCursor(xPos + sliders_text_delta, sliders_y0 - sliders_height);
  tft.setTextColor(SLIDER_TEXT);  
  tft.setTextSize(1);
  tft.println(String(sndSpeed) + " wpm");
  return;  
}

void displayTextTone() {
  int xPos = (selFreq - minFreq) * sliders_width / freqRange + sliders_xPos + sliders_pointer_width;
  const int text_width = 65;
  if (selFreq > maxFreq/2)
    xPos = xPos - text_width; 
  tft.setCursor(xPos + sliders_text_delta, sliders_y0 + sliders_yDelta - sliders_height);
  tft.setTextColor(SLIDER_TEXT);  
  tft.setTextSize(1);
  tft.println(String(selFreq) + " Hz");
  return;
}

void displayWordSpacingText() {
  int xPos = (selWordSpacing - minWordSpacing) * sliders_width / (maxWordSpacing - minWordSpacing) + sliders_xPos + sliders_pointer_width;
  const int text_width = 88;
  if (selWordSpacing > maxWordSpacing/2)
    xPos = xPos - text_width; 
  tft.setCursor(xPos + sliders_text_delta, sliders_y0 + sliders_yDelta + sliders_yDelta - sliders_height);
  tft.setTextColor(SLIDER_TEXT);  
  tft.setTextSize(1);
//tft.println(String(selWordSpacing) + "%");
  tft.println("(" + String(numSpaces) + ") " + WORD_SPACING_NAMES[numSpaces]);
  return;
  
}

int myGetX() {
  return (ts.getX() - X_MIN) * X_PIXELS / X_RANGE;
}

int myGetY() {
  return (ts.getY() - Y_MIN) * Y_PIXELS / Y_RANGE;
}

// Adapted from: Adafruit color oled test

void tftPrintText(String s) {
  tft.fillScreen(TFT_BG);
  tft.setCursor(0, 5);
  tft.setTextColor(TFT_ALT);  
  tft.setTextSize(2);
  tft.println(s);
}

void tftPrintText(String s, int16_t y) {
  tft.fillScreen(TFT_BG);
  tft.setCursor(0, y);
  tft.setTextColor(TFT_ALT);  
  tft.setTextSize(2);
  tft.println(s);
}

void tftPrintText(String s, int16_t x, int16_t y) {
  tft.fillScreen(TFT_BG);
  tft.setCursor(x, y);
  tft.setTextColor(TFT_ALT);  
  tft.setTextSize(2);
  tft.println(s);
}

void tftDisplayMenu() {
  tft.fillScreen(TFT_BG);
  for (int i=0; i<NUMOPT; i++) {
    displayButton(button_x0, i * (button_height+button_vert_separation) + button_y0, button_width, button_height, TFT_TXT, WHITE, optList[i], 2);
    tftDisplayMoreButton();
    tftDisplayReturnButton();
    tftDisplayEnglishWordsButton();
    tftDisplayRadioTermsButton();
  }
//tftMenuLevel=1;
}

void tftSlidersMenu() {
  tft.fillScreen(TFT_BG);
  displaySpeedSlider();
  displayToneSlider();
  displayWordSpacingSlider();
}

void tftDisplayMoreButton() {
  displayButton(button_x0, bottom_buttons_y, more_button_width, button_height, TFT_ALT, uColor(LT_GREY), "More...", 2);  
}

void tftDisplayReturnButton() {
  displayButton(return_button_x, bottom_buttons_y, return_button_width, button_height, TFT_ALT, uColor(LT_GREY), "Return", 2);
}

int tftOptionSelected() {
  int option = -1;
  for (int i=0; i<NUMOPT; i++) {
    if (tftTouchedInBox(button_x0, i * (button_height+button_vert_separation) + button_y0, button_width, button_height)) {
      option = i;
      break;
    }
  }
  return option;    
}

void tftPrintSelectedOption(int iOpt) {
  String s = "No option";
  if (v3_mode == 1) {
    s = "Common English words";
    tftMenuLevel = 2;
  }
  else if (v3_mode == 2) {
    s = "Ham radio terms";
    tftMenuLevel = 2;
  }
  else if (iOpt >=0) {
    s = optList[iOpt];
    tftMenuLevel = 2;
  }
  tftPrintText(" --> " + s, 100);
  tftDisplayReturnButton();
  if ((iOpt > 1) || (v3_mode > 0)) {
    delay(TWOSEC);
    moakTftCharCount = 0;
    tftInitTicker();
  }
}

void tftProcessSelectedOption(int iOpt) {
  if (v3_mode > 0) {
    tftClearV2Options();
    tftPrintSelectedOption(0);
  }
  else {
    tftSetSelectedOptionParameters(iOpt);
    currentOption = iOpt;
    tftPrintSelectedOption(iOpt);
  }
}

void tftSetSelectedOptionParameters(int iOpt) {
  // Cloned from pushbutton function processSelectButton()
  // Process each option type
  if (currentOption == iOpt)
    return;                    // Already set
  currentOption = iOpt;
  if (iOpt == 0) {             // Electronic keyer
    keyMode = true;
    straight = false;      
    ptMode = false;
    pcsMode = false;
    incNum = false;      
    practiceMode = false;
 }
  else if (iOpt == 1) {        // Straight key
    keyMode = true;
    straight = true;
    ptMode = false;
    pcsMode = false;
    incNum = false;      
    practiceMode = false;
  }
  else if (iOpt == 2) {        // 5-letter groups with display
    keyMode = false;
    ptMode = false;
    pcsMode = false;
    incNum = false;      
    practiceMode = false;
  }
  else if (iOpt == 3) {        // 5-character alphanumeric groups, with display
    keyMode = false;
    ptMode = false;
    pcsMode = false;
    practiceMode = false;
    incNum = true;
  }
  else if (iOpt == 4) {        // Pseudo-text, with display
    keyMode = false;
    ptMode = true;
    pcsMode = false;
    practiceMode = false;
    incNum = false;
  }
  else if (iOpt == 5) {        // Pseudo-text + numbers, with display
    keyMode = false;
    ptMode = true;
    pcsMode = false;
    practiceMode = false;
    incNum = true;
  }
  else if (iOpt == 6) {        // Pseudo call signs with display
    keyMode = false;
    ptMode = false;
    practiceMode = false;
    pcsMode = true;
  }
  else if (iOpt == 7) {        // Practice sending by keying-in displayed text
    keyMode = false;
    straight = false;          // Electronic key (paddle) only
    ptMode = false;
    pcsMode = false;
    practiceMode = true;
    incSound = false;          // Relevant only in sending practice modes
  }
  else if (iOpt == 8) {        // Practice sending by keying-in sounded and displayed text
    keyMode = false;
    straight = false;          // Electronic key (paddle) only
    ptMode = false;
    pcsMode = false;
    practiceMode = true;
    incSound = true;           // Sound pattern to copy
  }
  // Additional options go here
  
  // Regardless of whether option was changed or not -
  commandMode = false;
}

//  ISR and related experimentation

void flashLED() {
    digitalWrite(LED_TEST, LOW);
    delay(ONESEC/8);
    digitalWrite(LED_TEST, HIGH);
    delay(ONESEC/8);
    return;  
}

void flashLED(int n) {
  for (int i=0; i<n; i++)
    flashLED();
  delay(ONESEC/2);  
}

void touchISR() {
  flashLED();
  return;
}

void myDelay(unsigned long ms) {
/*
  delay(ms);
  return;
*/
  unsigned long endDelay = millis() + ms;
  while (millis() < endDelay) {
    if (ts.dataAvailable()) {  // Duplicate tftAbort() in-place to minimize overhead
      return; 
    } 
    delay(1);                  // Seems to help with touch detection (?)
  }
  return;
}


// EEPROM

void writeEEPROM(String s) {
  int sLen = s.length();
  if (sLen > EEPROM.length())
    return;
  for (int i=0; i<sLen; i++)
    EEPROM.write(i, s.charAt(i));  
}

String readEEPROM(int sLen) {
  if (sLen > EEPROM.length()) sLen = EEPROM.length();
  String s = "";
  for (int i=0; i<sLen; i++)
    s.concat((char) EEPROM.read(i));
  return s;
}

void eraseEEPROM(int len) {
  if (len > EEPROM.length()) len = EEPROM.length();
  for (int i=0; i<len; i++)
    EEPROM.write(i, 0);
}

void varsWriteEEPROM() {
  // Global variables SPEED, TONE FREQUENCY, WORD SPACING
  String s = String(sndSpeed);
  String t = String(selFreq);
  String v = String(selWordSpacing);
  while (s.length() < 2)
    s = ZERO + s;
  while (t.length() < 4)
    t = ZERO + t;
  while (v.length() < 2)
    v = ZERO + v;
  writeEEPROM("@@" + s + t + v);
}

void varsReadEEPROM() {
  String tmp = readEEPROM(10);
  if (tmp.substring(0, 2) == "@@") {
    sndSpeed = tmp.substring(2, 4).toInt();
    tftSetSpeed();   // dot and dash timing
    selFreq = tmp.substring(4, 8).toInt();
    selWordSpacing = tmp.substring(8, 10).toInt();
    tftNumSpaces();
  }
  return;
}

// Text display experimentation

void kilroy() {
  //  Do not erase display
  tft.setCursor(10, 100);
  tft.setTextColor(TFT_ALT);  
  tft.setTextSize(2);
  tft.println("Kilroy was here!");
  return;
}

void tftText(uint16_t x, uint16_t y, uint16_t sz, int n) {
  // Ascertain the number of (default font) characters of given size
  // that can be displayed horizontally.
  String s = String("abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz").substring(2, n);
  tft.setCursor(x, y);
  tft.setTextColor(BLACK);  
  tft.setTextSize(sz);
  tft.println(s);
  return;
}

// MOAK text display

void tftInitTicker() {
  for (int i=0; i<moakTftROWS; i++) {
    for (int j=0; j<moakTftCOLS; j++) {
       moakTftText[i][j] = 0; 
    }
  }
//tft.fillRect(moakTftTextXinset, moakTftTextYdelta, X_PIXELS-moakTftTextXinset, moakTftTextYdelta * (moakTftROWS+1), TFT_BG);
  tft.fillRect(0, moakTftTextYdelta, X_PIXELS, moakTftTextYdelta * (moakTftROWS+1), TFT_BG);
  tft.drawRect(moakTftTextXinset/2, moakTftTextYdelta/2, X_PIXELS-moakTftTextXinset, moakTftTextYdelta * (moakTftROWS+1), TFT_ALT);
  tft.setCursor(moakTftTextXinset, moakTftTextYdelta);
  tft.setTextColor(moakTftOutputColor);
  tft.setTextSize(moakTftTextSize);
}

void tftTickerAddCharacter(char c) {    // Analog of LCD function: tickerAddCharacter(char c)
  int maxCount = moakTftROWS * moakTftCOLS;
  int row, col;
  if (tftAbort()) return;
//if (not displayCode) return;          // Disable for testing (Set in LCD context and not yet integrated with TFT)
//if (commandMode) return;              // Not relevant in TFT environment (not updated properly)
  if (practiceMode) return;             // Do not duplicate display in practice mode
  if (++moakTftCharCount > maxCount) {
/*
    // Shift up
    for (int i=0; i < (maxCount-moakTftCOLS); i++) {
      row = i / moakTftCOLS;
      col = i % moakTftCOLS;
      moakTftText[i][j] = moakTftText[i+1][j];
    }
*/
    // Clear and restart at top (simpler than vertical scrolling)
    moakTftCharCount = 1;
  }
  if (moakTftCharCount == 1) {
    tftInitTicker();    
  }
  row = moakTftCharCount / moakTftCOLS;
  col = moakTftCharCount % moakTftCOLS;
  if (row > 0 && col == 1) {            // Text Box alignment
    tft.setCursor(moakTftTextXinset,moakTftTextYdelta * (row+1));
  }
  moakTftText[row][col] = c;
  tft.print(c);
}

void displayTFT(String line1, String line2) {
  // TFT analog of: void displayLCD(String line1, String line2)
  // Variable row is not relevant
  tftInitTicker();
  tft.setCursor(moakTftTextXinset, moakTftTextYdelta);
  tft.print(line1);
  tft.setCursor(moakTftTextXinset, 2*moakTftTextYdelta);
  tft.print(line2); 
}

void tftAppend(char c) {
  // TFT analog of lcd.print(character)
  // Display character using current settings (current cursor position)
  tft.print(c);
}

// Cosmetics

String tftFormat(String s) {
  String sL, sR;
  if (s.length() == 0)
    return s;
  while (s.length() < moakTftCOLS)
    s.concat(SPACE);
  int rightmostSpacePos;
  for (int i=moakTftCOLS-1; i>=0; i--)
    if (s.charAt(i) == SPACE) {
      rightmostSpacePos = i;
      break;
    }
  sL = s.substring(0, rightmostSpacePos);
  sR = s.substring(rightmostSpacePos+1);
  while (sL.length() < moakTftCOLS)
    sL.concat(SPACE);
  return sL + tftFormat(sR);
}

// TFT timing

void tftSetSpeed() {
  dotTime =  timeConstant / sndSpeed;
  dashTime = dotTime + dotTime + dotTime;
  charTime = dashTime;
  wordTime = dashTime + dashTime + dotTime;  
}

// New options MOAK version 3.x

String englishWord() {
  // Pseudo-random word selected from 1000 most common words in written English
  // Probability = occurrence frequency in written English
  long N = sizeof(WORDWEIGHT) / sizeof(WORDWEIGHT[0]);  // WORDLIST and WORDWEIGHT arrays have the same size
  long nwt = random(WORDWEIGHT[N-1]);
  int ndx;
  for (ndx=0; ndx<N; ndx++)
    if (nwt < WORDWEIGHT[ndx])
      break;
  return WORDLIST[ndx];
}

String rndews() {
  String s = "";
  int sl = random(MAXSLEN) + MINSLEN;
  for (int i=1; i<sl; i++) {
    s.concat(englishWord() + SPACE);
  }
  s.concat(englishWord() + PERIOD);
  if (tftPresent)
    return tftFormat(s);
  else
    return s;  
}

String radioTerm() {
  return RADIOTERM[random(NUMTERMS)];
}

String rndrts() {
  String s = "";
  int sl = random(NUMTERMS) + MINSLEN;
  for (int i=1; i<sl; i++) {
    s.concat(radioTerm() + SPACE);
  }
  s.concat(radioTerm() + PERIOD);
  if (tftPresent)
    return tftFormat(s);
  else
    return s;  
}

void tftDisplayEnglishWordsButton() {
  tft.fillRoundRect(button_x0+button_width+button_hor_separation, button_y0, right_buttons_width, 3*button_height + 2*button_vert_separation, button_radius, GREEN);
  tft.drawRoundRect(button_x0+button_width+button_hor_separation, button_y0, right_buttons_width, 3*button_height + 2*button_vert_separation, button_radius, BLACK);
  tft.setTextColor(TFT_TXT);
  tft.setTextSize(moakTftTextSize);
  tft.setCursor(button_x0+button_width+button_hor_separation+5, button_y0+2);
  tft.print(" 1000");
  tft.setCursor(button_x0+button_width+button_hor_separation+5, button_y0+button_height+button_vert_separation);
  tft.print("common");
  tft.setCursor(button_x0+button_width+button_hor_separation+5, button_y0+2*(button_height+button_vert_separation));
  tft.print("words!");
}

void tftDisplayRadioTermsButton() {
  tft.fillRoundRect(button_x0+button_width+button_hor_separation, button_y0 + 4*button_height + 3*button_vert_separation + 2, right_buttons_width, 3*button_height + 2*button_vert_separation, button_radius, GREEN);
  tft.drawRoundRect(button_x0+button_width+button_hor_separation, button_y0 + 4*button_height + 3*button_vert_separation + 2, right_buttons_width, 3*button_height + 2*button_vert_separation, button_radius, BLACK);
  tft.setTextColor(TFT_TXT);
  tft.setTextSize(moakTftTextSize);
  tft.setCursor(button_x0+button_width+button_hor_separation+5, button_y0 + 4*(button_height + button_vert_separation) + 2);
  tft.print(" Ham");
  tft.setCursor(button_x0+button_width+button_hor_separation+5, button_y0 + 5*(button_height+button_vert_separation));
  tft.print("radio");
  tft.setCursor(button_x0+button_width+button_hor_separation+5, button_y0 + 6*(button_height+button_vert_separation));
  tft.print("terms");
}

boolean tftEnglishWordsButtonPressed() {
  return tftTouchedInBox(button_x0+button_width+button_hor_separation, button_y0, right_buttons_width, 3*button_height + 2*button_vert_separation);
}

boolean tftRadioTermsButtonPressed() {
  return tftTouchedInBox(button_x0+button_width+button_hor_separation, button_y0 + 4*button_height + 3*button_vert_separation + 2, right_buttons_width, 3*button_height + 2*button_vert_separation);
}

void tftClearV2Options() {
    keyMode = false;    
    ptMode = false;
    pcsMode = false;
    practiceMode = false;
}

void tftSetDefaultMode() {
    keyMode = true;
    ptMode = false;
    pcsMode = false;
    practiceMode = false;
    incNum = false;      
}

// Adapted from: https://afterthoughtsoftware.com/posts/convert-rgb888-to-rgb565

uint16_t uColor(uint8_t RGB[]) {
  
    uint16_t b = (RGB[2] >> 3) & 0x1f;
    uint16_t g = ((RGB[1] >> 2) & 0x3f) << 5;
    uint16_t r = ((RGB[0] >> 3) & 0x1f) << 11;

    return (uint16_t) (r | g | b); 
}
