/*
	Communication between computers (using Morse Code) requires EWD.js and a MUMPS database
    (http://ec2.mgateway.com/ewd/ws/index.html).
    
    Without Node.js and EWD.js, etc., keying-in and playing back Morse code works in selected
    browsers, with communications functions disabled.
    
	Lloyd Milligan / himself@lloydm.net (April 2016)
*/  

	var IE = false;

    if (detectIE()) {
      IE = true;
	  if (!window.confirm("In order to generated a tone, this page requires a feature\nthat is not supported by this browser.")) window.close();
    }

	var firefox = false;

	if(navigator.userAgent.toLowerCase().indexOf('firefox') > -1) {
      firefox = true;
	  document.getElementById('playText').disabled = true;
    }

//  Global definitions

    var ele = '';			// Current/last element (debug usage)
	var volume = 0.4;		// Default - Range is 0. to 1.
	var frequency = 440;	// Hertz
	var MUTE = 0.;
	var keyDownImage = 'Images/key-nzart.org-down-rev.jpg';
	var keyUpImage = 'Images/key-nzart.org-up-rev.jpg';
	var redLight = 'Images/traffic-light-red-dan-ge-01-cropped-small-OpenClipArt.org.jpg';
	var yellowLight = 'Images/traffic-light-yellow-dan-01-cropped-small-OpenClipArt.org.jpg';
	var greenLight = 'Images/traffic-light-green-dan--01-cropped-small-OpenClipArt.org.jpg';
	var comRed = '#ff3333';
	var comGreen = '#00cc00';


//  Discrimination thresholds

	var d1 = 100;	// 1 x dot - will be revalued by short or long test
	var d2 = 200;	// 2 x dot - will be revalued by short or long test
    var d3 = 300;	// 3 x dot - will be revalued by short or long test
	var d4 = 400;	// 4 x dot - will be revalued by short or long test
	var d5 = 500;	// 5 x dot - wlll be revalued by short or long test
	var calibrated = false;

//  Other Timing

	var sndSpeed = 12;		// 12 WPM corresponds to dot duration of 100 ms
	var rcvSpeed = 12;		// Reset based on user selection
	var spaceRatio = 1;		// Multiplier for duration of spaces (playback/receive only)

	var firstKeyDown = 0;
	var lastKeyUp;
	var timeKeyDown;
	var timeKeyUp = Date.now();

	var interval = 2000;	//milliseconds
	var intervalID = window.setInterval(periodicActions, interval);

//  Buffers and indexes
 
	var tBuf = [];	// Array of times
	var cBuf = [];	// Array of characters as code elements
	var mBuf = [];	// Message buffer (text array)
	var tNdx = 0;
	var cNdx = 0;
	var mNdx = 0;

//  Audio

	if (!IE) {

    var context = new AudioContext();
    var osc = context.createOscillator();
    osc.frequency.value = frequency;
    var vol = context.createGain();

    vol.gain.value = MUTE;
    osc.connect(vol); // connect osc to vol
    vol.connect(context.destination); // connect vol to context distination
    osc.start(0); // start immediately
    }

//  Cache

	var codeTextId = document.getElementById('codeText');
	var keyId = document.getElementById('keyImage');
	var statusMessageId = document.getElementById('messageInput');
	var sigLightImageId = document.getElementById('trafficLightImage');
	var sigLightCaptionId = document.getElementById('trafficLightCaption');
	var resultAreaId = document.getElementById('resultTextArea');
	var ewdLabelId = document.getElementById('ewdLabel');
	var myCallId = document.getElementById('myCall');
	var hisCallId = document.getElementById('hisCall');
	var halfDuplexId = document.getElementById('halfDuplexLbl');
	var fullDuplexId = document.getElementById('fullDuplexLbl');
    var volumeId = document.getElementById('volSlider');
	var frequencyId = document.getElementById('freqNumber');
	var rcvSpeedId = document.getElementById('rcvSpeedNumber');
	var spaceRatioId = document.getElementById('spaceRatioId');

//  Communication (EWD.js registration will be commented-out in non-comm version)

	var comReady = false;
	var cwConnected = false;
	var cwTest = false;
	var halfDuplex = false;
	var fullDuplex = false;
	var remoteStation = '';
	var myCall = '';
	var hisCall = '';
	var cqSent = false;
	var cqAnswered = false;
	var uid = randomString(8, '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ');


EWD.application = {
	name: 'my_cw',

		onMessage: {

            cwMsg : function(messageObj) {
              var text = messageObj.message.data[0];
              if (typeof(text) == 'undefined') return;
              if (error(text)) return;
              var replyCode = piece(text,"^");
              var message = piece(text, "^", 2);
              if (replyCode == 0)		//QRK reply
                playNice(message.slice(0));
              if (replyCode == 1) {		//Successfully registered (QRZ)
                myCall = piece(message, ' ', 2);
                myCallId.value = myCall;
              }
              else if (replyCode == 2) {//Successfully unregistered (QRT)
                myCall = '';
                hisCall = '';
                myCallId.value = myCall;
                hisCallId.value = hisCall;
                cqSent = false;			//Reset connection
                cqAnswered = false;
                halfDuplexId.style.backgroundColor = comRed;
                fullDuplexId.style.backgroundColor = comRed;
              }
              if ((replyCode > 0) &&
                  (replyCode < 3))	{	// Normal status message from server
                displayMessage(piece(text,"^",2));
                return;
              }
              if (replyCode == 11) {	//QBM reply
                if (piece(message, ' ', 2) == 'DE')	{	//From previously unknown station
                  var from = piece(message,' ', 3);
                  if (from != '') {
                    hisCall = from;
                    hisCallId.value = hisCall;
                  }
                  halfDuplex = true;
                  halfDuplexId.style.backgroundColor = comGreen;
                }
                playNice(message.slice(0));
                return;
              }
              if (replyCode == 12) {
                halfDuplex = true;
                halfDuplexId.style.backgroundColor = comGreen;
                hisCall = message;
                hisCallId.value = hisCall;
                return;
              }
              return;
     	   }


			// Insert additional reply message handlers here

		}

};

EWD.sockets.log = true; // Change to false when development is complete

EWD.onSocketsReady = function() {
    comReady = true;
    ewdLabelId.style.backgroundColor = comGreen;
};


//  Event handlers

    function codeKeyDown() {
      var now, sp;
      if (!IE)
        toneOn();
      now = Date.now();
      timeKeyDown = now;
      if (tNdx == 0)
        firstKeyDown = now
       else {
         sp = now - timeKeyUp;
         if (calibrated) {
           if (sp > d5) {	// End of word
             if (typeof(cBuf[cNdx]) == "undefined")
               cBuf[cNdx] = ele
             else
               cBuf[cNdx] = cBuf[cNdx] + ele;
             mBuf[mNdx++] = codeToChar(cBuf[cNdx++]);
             codeTextId.value = codeTextId.value + mBuf[mNdx-1] + ' ';
             mBuf[mNdx++] = ' ';
             // displayMessage('EOW');
           }
           else if (sp > d2) {	// End of character
             if (typeof(cBuf[cNdx]) == "undefined")
               cBuf[cNdx] = ele
             else
               cBuf[cNdx] = cBuf[cNdx] + ele;
             mBuf[mNdx++] = codeToChar(cBuf[cNdx++]);
             codeTextId.value = codeTextId.value + mBuf[mNdx-1];
             // displayMessage('EOC');
           }
           else {	// Element
             if (typeof(cBuf[cNdx]) == "undefined")
               cBuf[cNdx] = ele
             else
               cBuf[cNdx] = cBuf[cNdx] + ele;
             // displayMessage('EOE');
           }
         }
       }
      // displayMessage('Key down');
      keyId.src = keyDownImage;
      return;
};

    function codeKeyUp() {
      if (!IE)
        toneOff();
      timeKeyUp = Date.now();
      var ms = timeKeyUp - timeKeyDown;
      tBuf[tNdx++] = ms;
      if (calibrated) {	// Key up always terminates either a dot or a dash
        if (ms < d2)
          ele = '.'
        else
          ele = '-';
      }
        
      // displayMessage(ms + ' ms' + ' ' + ele);	// Debug
      keyId.src = keyUpImage;
      return;      
};

	function btnReset() {
      // Key-up
      if (!IE) vol.gain.value = MUTE;
      keyId.src = keyUpImage;
      // Initialize and require re-calibration
      initBuffers();
      calibrated = false;
      // Clear textual displays
	  sendBtnClearMessage();
	  btnClearText();
      btnClearResult();
      // Show calibration status
      sigLightImageId.src = redLight;
      sigLightCaptionId.innerHTML = 'Calibration Status';
	  rcvSpeedId.disabled = true;	// Until calibrated
      rcvSpeedId.value = '';	// Can be adjusted
      return;
};

	function btnShortTest() {
      // Ascertain code speed and set discrimination thresholds
      var dur = timeKeyUp - firstKeyDown;
      d1 = Math.round(dur / 33);
      d2 = d1 + d1;
      d3 = d1 + d2;
      d4 = d2 + d2;
      d5 = d4 + d1;
      sndSpeed = Math.round(396000 / dur) / 10;
      if (tNdx == 12) {
        calibrated = true;
      	displayMessage('Short test: Duration = ' + dur + ' milliseconds, d2 = ' + d2 + ', d5 = ' + d5);
        sigLightImageId.src = greenLight;
        sigLightCaptionId.innerHTML = 'Calibration ' + sndSpeed + ' WPM';
      }
      else {
        displayMessage("Error: Please retry keying 3 V's exactly, no more, no less!", "red");
        sigLightImageId.src = yellowLight;
      }
      initBuffers();
      rcvSpeed = sndSpeed;	// Default receive speed
      if (firefox) return;	// Receive speed component is not relevant
	  rcvSpeedId.disabled = false;	// Enable after calibration
      rcvSpeedId.value = parseFloat(rcvSpeed);	// Can be adjusted
      return;
};

	function btnLongTest() {
      // Ascertain code speed and set discrimination thresholds
      var dur = timeKeyUp - firstKeyDown;
      d1 = Math.round(dur / 143);
      d2 = d1 + d1;
      d3 = d1 + d2;
      d4 = d2 + d2;
      d5 = d4 + d1;
      var dur = timeKeyUp - firstKeyDown;
      sndSpeed = Math.round(1716000 / dur) / 10;
      if (tNdx == 42) {
        calibrated = true;
      	displayMessage('Long test: Duration = ' + dur + ' milliseconds, d2 = ' + d2 + ', d5 = ' + d5);
        sigLightImageId.src = greenLight;
        sigLightCaptionId.innerHTML = 'Calibration ' + sndSpeed + ' WPM';
      }
      else {
        displayMessage("Error: Please retry keying the word PARIS 3 times exactly, no more, no less!", "red");
        sigLightImageId.src = yellowLight;
      }
      initBuffers();
      rcvSpeed = sndSpeed;	// Default receive speed
      if (firefox) return;	// Receive speed component is not relevant
	  rcvSpeedId.disabled = false;	// Enable after calibration
      rcvSpeedId.value = parseFloat(rcvSpeed);	// Can be adjusted
      return;
};

	function btnShowText() {
      if (!calibrated) {
        displayMessage("Error: You must calibrate first! (Click 'Reset')","red");
        return;
      }
      var message = '';
      // Store last element
      if (typeof(cBuf[cNdx]) == "undefined")
        cBuf[cNdx] = ele
      else
        cBuf[cNdx] = cBuf[cNdx] + ele;
      mBuf[mNdx] = codeToChar(cBuf[cNdx]);

      for (var i = 0; i < mBuf.length ; i++)
        message = message + mBuf[i];
        // message = message + cBuf[i] + '|';
      codeTextId.value = message;
      resultAreaId.value += message + '\n';
      initBuffers();
      return;
};

	function btnClearText() {
      if (document.getElementById('pwd').value == 'test' && calibrated) {
        btnPlayMessageText();
        return;
      }
      codeTextId.value = '';
      return;
};

	function btnClearResult() {
      resultAreaId.value = '';
      return;
};

	function adjustVol() {
	  volume = volumeId.value / 100;
	  return;
};

	function adjustFreq() {
	  frequency = frequencyId.value;
          osc.frequency.value = frequency;
	  return false;
};

	function adjustReceiveSpeed() {
	  rcvSpeed = rcvSpeedId.value;
	  return false;
};

	function adjustSpaceRatio() {
	  spaceRatio = spaceRatioId.value;
      if (spaceRatio == 1) displayMessage('Normal character spacing selected.')
      else displayMessage('Character spacing = ' + spaceRatio + ' times normal.');
	  return false;
};

	function btnPlayMessageText() {
      var str = codeTextId.value.slice();
      if (calibrated) {
        pause(d1);	// In order for str to have '#codeText'
        playNice(str);
      }
      else
        displayMessage('Please calibrate first!','red');
      return;
};

	function periodicActions() {
      if (typeof(timeKeyDown) == 'undefined') return;	// Nothing has happened yet
      if (!calibrated) return;	// No keyed data to process
	  if (Date.now() - (timeKeyUp > timeKeyDown ? timeKeyUp : timeKeyDown) > interval) {
		// Update Code Text display
        if (mNdx > 0) {
          btnShowText();	// Simply replace
          routeMessage(codeTextId.value);	// If any...
        }
        else processQBM();	// Poller
	  }
      return;
};

	function routeMessage(s) {
      // Parameter s is a copy of codeTextId.value, usually junk
      // but sometimes an initial transmission and sometimes a reply to
      // a received message.  This function ascertains which of these
      // cases applies, and routes the text value accordingly.
      if (!calibrated || !comReady)
        return;	// Fundamental
      if (cwConnected && remoteStation != '') {
        sendReply(s);
        return;
      }
      var x = piece(s, ' ');
      var from = piece(s, ' ', 2, 3);	//DE <call>, if part of 2-way communication
      if (x == 'QRK')
        sendTestRequest(s)
      else if (x == 'QRZ')
        processQRZ(s)
      else if (x == 'QRT')
        processQRT(s)
      else if (x == 'CQ')
        processCQ(s)
      else if ((myCall != '') && (from == 'DE ' + myCall))
        processCOMM(s)
      else if (x == 'R')
        processReply(s)
	  // If fall through here: Unsupported request or junk
      return;
};

	function sendTestRequest(s) {	// Ping
      // No change in connection status
      sendInitialMessage(s);
      return;
};

	function processQRZ(s) {
      // Register user's handle (or replacement handle)
      sendInitialMessage(s);
      return;
};

	function processQRT(s) {
      // Unregister user's handle
      sendInitialMessage(s);
      return;
};

	function processQBM(s) {
      // Poll for incoming message
      if (myCall == '') return;	// Current registration required!
      sendInitialMessage('QBM DE ' + myCall);
      return;
};

	function processCQ(s) {
      // Message to all registered listeners (except self)
	  sendBtnClearMessage();
      sendInitialMessage(s);
      cqSent = true;
      return;
};

	function processCOMM(s) {
      // Message to a specified recipient (another station)
	  sendBtnClearMessage();
      sendInitialMessage(s);
      return;
};

	function processReply(s) {
      // First character should be 'R' followed by <space>
      // Substitute TO DE FROM if valued
      // Else, cannot process naked reply
	  sendBtnClearMessage();
      if ((myCall == '') || (hisCall == '')) return;
      sendInitialMessage(hisCall + ' DE ' + myCall + ' ' + s);
      return;
};


//  EWD.js message senders

	function sendInitialMessage(s) {
      EWD.sockets.sendMessage({
        type: "cwMsg",
        params: {
          data: s,
          UID: uid
        }
    });      
      return;
};



//  Utilities

    function error(str) {
      // If str is an error report, return TRUE and display message
      // Else, return false
          if (typeof str == 'undefined')
      str = '';
      var code = piece(str, '^');
      var msg = piece(str, '^', 2);
      if (code == '-1') {
        displayMessage(msg, 'red');
        return true;
      }
      return false;
};

	function displayMessage(msg, textColor) {
      if (typeof(textColor) == "undefined") textColor = '';
      statusMessageId.style.color = textColor;
      statusMessageId.value = msg;
};

	function appendMessage(msg, textColor) {
      if (typeof(textColor) == "undefined") textColor = '';
      statusMessageId.style.color = textColor;
      statusMessageId.value = statusMessageId.value + msg;
};

	function sendBtnClearMessage() {
      statusMessageId.value = '';
};

	function initBuffers() {
      tBuf.length = 0, tNdx = 0;
      cBuf.length = 0, cNdx = 0;
      mBuf.length = 0, mNdx = 0;
      return;
};

	function setSpeed(wpm) {
      if (isNaN(wpm)) return;
      if ((wpm < 1) || (wpm > 20)) return;
      d1 = 1200 / wpm;	// dot duration in milliseconds
      d2 = d1 + d1, d3 = d2 + d1, d4 = d2 + d2, d5 = d4 + d1;
      return
};

    function toneOn() {
      vol.gain.value = volume;
      return;
};

    function toneOff() {
      vol.gain.value = MUTE;
      return;
};

    function makeDot() {
	  toneOn();
      pause(d1);
      toneOff();
      eleSpace();
      return;
};

    function makeDash() {
	  toneOn();
      pause(d3);
      toneOff();
      eleSpace();
      return;
};

    function eleSpace() {
      pause(d1);
      return;
};

    function chrSpace() {
      // Character space is d3. This function is character space - element space (included in element generation)
      pause(d2 * spaceRatio);
      return;
};

    function wordSpace() {
      // Word space is d7. This function is word space - character space (included in character generation)
      pause(d4 * spaceRatio);
      return;
};

    function doNothing() {
      return;
};

    function playCode(character) {
      var code = charToCode(character);
      var ele = '';
      for (var j = 0; j < code.length; j++) {
        ele = code.substring(j, j+1);
        if (ele == '.') makeDot()
        else if (ele == '-') makeDash()
        else break;
      }
      if (ele == ' ' || (ele == '*')) wordSpace()
      else chrSpace();
      return;
};

	function playNice(str) {	//Wraps playText at possibly slower speed
      setSpeed(rcvSpeed);
      playText(str);
      setSpeed(sndSpeed);
      return;
};

	function playText(str) {
      if (typeof(str) == 'undefined') return;
      if (str.length == 0) return;
      str = str.toUpperCase();
      
      var a, b, c;
      a = piece(str, '|'), b = piece(str, '|', 2), c= piece(str, '|', 3, 99);
      if (b.length > 0) {
        playText(a);
        playCode('|' + b + '|');
        playText(c);
        return;
      }
      
      if (str == "undefined") str = '';
      for (var j = 0; j < str.length; j++)
        playCode(str.charAt(j));
      return;
};

	function codeToChar(code) {
      switch(code) {
          case ' ':
              return ' ';
          case '.-':
              return 'A';
          case '-...':
              return 'B';
          case '-.-.':
              return 'C';
          case '-..':
              return 'D';
          case '.':
              return 'E';
          case '..-.':
              return 'F';
          case '--.':
              return 'G';
          case '....':
              return 'H';
          case '..':
              return 'I';
          case '.---':
              return 'J';
          case '-.-':
              return 'K';
          case '.-..':
              return 'L';
          case '--':
              return 'M';
          case '-.':
              return 'N';
          case '---':
              return 'O';
          case '.--.':
              return 'P';
          case '--.-':
              return 'Q';
          case '.-.':
              return 'R';
          case '...':
              return 'S';
          case '-':
              return 'T';
          case '..-':
              return 'U';
          case '...-':
              return 'V';
          case '.--':
              return 'W';
          case '-..-':
              return 'X';
          case '-.--':
              return 'Y';
          case '--..':
              return 'Z';
          case '.----':
              return '1';
          case '..---':
              return '2';
          case '...--':
              return '3';
          case '....-':
              return '4';
          case '.....':
              return '5';
          case '-....':
              return '6';
          case '--...':
              return '7';
          case '---..':
              return '8';
          case '----.':
              return '9';
          case '-----':
              return '0';
          case '.-.-.-':
              return '.';
          case '--..--':
              return ',';
          case '..--..':
              return '?';
          case '-..-.':
              return '/';
          case '.--.-.':
              return '@';
          case '-...-':
              return '|=|';
          case '.-.-.':
              return '|AR|';
          case '-...-.-':
              return '|BK|';
          case '-...-':
              return '|BT|';
          case '-.--.':
              return '|KN|';
          case '...-.-':
              return '|SK|';
          case '...-.':
              return '|SN|';
        case '...---':          
              return '|SO|';
        case '---...':          
              return '|OS|';
        case '...---...':          
              return '|SOS|';
          case '........':
              return '<BS>';	// Error / Backspace
          default:
              return '*';
    }
};

	function charToCode(character) {
      switch(character) {
          case ' ':
              return ' ';
          case 'A':
              return '.-';
          case 'B':
              return '-...';
          case 'C':
              return '-.-.';
          case 'D':
              return '-..';
          case 'E':
              return '.';
          case 'F':
              return '..-.';
          case 'G':
              return '--.';
          case 'H':
              return '....';
          case 'I':
              return '..';
          case 'J':
              return '.---';
          case 'K':
              return '-.-';
          case 'L':
              return '.-..';
          case 'M':
              return '--';
          case 'N':
              return '-.';
          case 'O':
              return '---';
          case 'P':
              return '.--.';
          case 'Q':
              return '--.-';
          case 'R':
              return '.-.';
          case 'S':
              return '...';
          case 'T':
              return '-';
          case 'U':
              return '..-';
          case 'V':
              return '...-';
          case 'W':
              return '.--';
          case 'X':
              return '-..-';
          case 'Y':
              return '-.--';
          case 'Z':
              return '--..';
          case '1':
              return '.----';
          case '2':
              return '..---';
          case '3':
              return '...--';
          case '4':
              return '....-';
          case '5':
              return '.....';
          case '6':
              return '-....';
          case '7':
              return '--...';
          case '8':
              return '---..';
          case '9':
              return '----.';
          case '0':
              return '-----';
          case '.':
              return '.-.-.-';
          case ',':
              return '--..--';
          case '?':
              return '..--..';
          case '/':
              return '-..-.';
          case '@':
              return '.--.-.';
          case '=':
              return '-...-';
          case '|AR|':
              return '.-.-.';
          case '|BK|':
              return '-...-.-';
          case '|BT|':
              return '-...-';
          case '|KN|':
              return '-.--.';
          case '|SK|':
              return '...-.-';
          case '|SN|':
              return '...-.';
          default:
              return '........';
    }
};

	  var piece = function(p1, p2, p3, p4) {	// $PIECE - from mString
        if (typeof p1 == "undefined") return '';
        if (typeof p2 == "undefined") return '';
        if (typeof p3 == "undefined") p3 = 1;
        if (typeof p4 == "undefined") p4 = p3;
        // Special cases
        if (p3 > p4) return '';
        if (p1.indexOf(p2) < 0) {
            if (p3 > 1) return '';
            else return p1;
        }
        // Standard case
        var i = 0, strt = 0, result = '';
        for (var j = 0; j < p1.length; j += 1) {
            var ii = i;
            i = p1.indexOf(p2, i) + p2.length;
            if (j + 1 == p3) {
              strt = ii;
              result = p1.substring(strt);
	        }
            if (j == p4) {
			  result = result.substring(0, ii - strt - p2.length);
        	  break;
	    	}
    	    if (i <= ii) break;    // Fewer than p4 instances of p2 in p1
        }
        return result;
      };


//  From other sources

    function detectIE() {	//Adapted from - http://codepen.io/gapcode/pen/vEJNZN
      
      var ua = window.navigator.userAgent;

      // Test values; Uncomment to check result &

      if (ua.indexOf('Trident') > 0)	// IE 10 or 11
        return true;


      if (ua.indexOf('MSIE ') > 0)		// Older versions
        return true;

      // other browser
      return false;
};

      function pause(ms) { //	By Pavel Bakhilau - See http://www.sean.co.uk/a/webdesign/javascriptdelay.shtm
      ms += new Date().getTime();
      while (new Date() < ms){}	
}; 

//  From: http://stackoverflow.com/questions/10726909/random-alpha-numeric-string-in-javascript

	  function randomString(length, chars) {
        var result = '';
        for (var i = length; i > 0; --i) result += chars[Math.floor(Math.random() * chars.length)];
        return result;
};
